/*
 * (c) 2023-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Hilfsfunktionen fuer die grafische Oberflaeche
 */

package jkcload.ui;

import java.awt.Component;
import java.awt.Dialog;
import java.awt.EventQueue;
import java.awt.Frame;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.Image;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.datatransfer.DataFlavor;
import java.awt.dnd.DnDConstants;
import java.awt.dnd.DropTargetDragEvent;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import javax.sound.sampled.AudioFileFormat;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JSlider;
import javax.swing.JTable;
import javax.swing.JViewport;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;
import jkcload.Main;
import jkcload.audio.AudioUtil;


public class UIUtil
{
  public static final String ITEM_CLOSE         = "Schlie\u00DFen";
  public static final String STATUS_READY       = "Bereit";
  public static final String TITLE_CONFIRMATIOn = "Best\u00E4tigung";

  private static final String PROP_WINDOW_X          = "window.x";
  private static final String PROP_WINDOW_Y          = "window.y";
  private static final String PROP_WINDOW_WIDTH      = "window.width";
  private static final String PROP_WINDOW_HEIGHT     = "window.height";
  private static final String PROP_TABLE_COL_COUNT   = "table.col.count";
  private static final String PROP_TABLE_COL_X_NAME  = "table.col.%d.name";
  private static final String PROP_TABLE_COL_X_WIDTH = "table.col.%d.width";
  private static final String PROP_AUDIO_PLAY_MIXER
				= Main.PROP_PREFIX + "audio.play.mixer.name";
  private static final String PROP_AUDIO_PLAY_VOLUME
				= Main.PROP_PREFIX + "audio.play.volume";


  private static final String[] iconImageResources = {
					"/images/jkcload_20x20.png",
					"/images/jkcload_24x24.png",
					"/images/jkcload_32x32.png",
					"/images/jkcload_48x48.png" };

  private static Image       largestIconImage = null;
  private static List<Image> iconImages       = null;
  private static FileFilter  audioFileFilter  = null;


  public static void acceptDragInCaseOfFileTransfer( DropTargetDragEvent e )
  {
    DataFlavor[] dataFlavors = e.getCurrentDataFlavors();
    if( dataFlavors != null ) {
      for( DataFlavor dataFlavor : dataFlavors ) {
	if( dataFlavor.equals( DataFlavor.javaFileListFlavor ) ) {
	  e.acceptDrag( DnDConstants.ACTION_COPY );
	  break;
	}
      }
    }
  }


  public static JButton createImageButton( String resource, String text )
  {
    JButton btn = null;
    Image   img = null;
    URL     url = Main.class.getResource( resource );
    if( url != null ) {
      img = Toolkit.getDefaultToolkit().getImage( url );
    }
    if( img != null ) {
      btn = new JButton( new ImageIcon( img ) );
      if( !text.isEmpty() ) {
	if( Character.isLetter( text.charAt( 0 ) ) ) {
	  btn.setToolTipText( text );
	}
      }
    } else {
      btn = new JButton( text );
    }
    return btn;
  }


  public static void fireShowErrorDnDOperationFailed(
						Window    owner,
						Exception ex )
  {
    EventQueue.invokeLater(
		()->showErrorMsg(
			owner,
			"Drag&Drop-Operation fehlgeschlagen",
			ex ) );
  }


  public static void fireShowErrorMsg(
				Window    owner,
				String    msg,
				Exception ex )
  {
    EventQueue.invokeLater( ()->showErrorMsg( owner, msg, ex ) );
  }


  public static FileFilter getAudioFileFilter()
  {
    if( audioFileFilter == null ) {
      try {
	Set<String> ext  = AudioUtil.getFileExtToAudioFileTypeMap().keySet();
	int         nExt = ext.size();
	if( nExt > 0 ) {
	  String[] extensions = ext.toArray( new String[ nExt ] );
	  Arrays.sort( extensions );
	  StringBuilder buf = new StringBuilder( 128 );
	  for( String fileExt : extensions ) {
	    if( buf.length() > 0 ) {
	      buf.append( "; *." );
	    } else {
	      buf.append( "Audiodateien (*." );
	    }
	    buf.append( fileExt );
	  }
	  buf.append( ')' );
	  audioFileFilter = new FileNameExtensionFilter(
						buf.toString(),
						extensions );
	}
      }
      catch( ArrayStoreException ex ) {}
    }
    return audioFileFilter;
  }


  public static AudioFileFormat.Type getAudioFileType( String fileName )
							throws IOException
  {
    AudioFileFormat.Type fileType = null;
    if( fileName != null ) {
      int idx = fileName.lastIndexOf( File.separator );
      if( idx >= 0 ) {
	fileName = fileName.substring( idx );
      }
      idx = fileName.lastIndexOf( '.' );
      if( (idx >= 0) && ((idx + 1) < fileName.length()) ) {
	fileType = AudioUtil.getAudioFileTypeByExtension(
					fileName.substring( idx + 1 ) );
      }
    }
    if( fileType == null ) {
      throw new IOException( "Das durch die Dateiendung angegebene"
			+ " Dateiformat\n"
			+ "wird nicht unterst\u00FCtzt." );
    }
    return fileType;
  }


  public static String getAudioPlayMixerName()
  {
    return Main.getProperty( PROP_AUDIO_PLAY_MIXER );
  }


  public static float getAudioPlayVolume()
  {
    float  v = -1F;
    String s = Main.getProperty( PROP_AUDIO_PLAY_VOLUME );
    if( s != null ) {
      try {
	v = Float.parseFloat( s );
      }
      catch( NumberFormatException ex ) {}
    }
    if ( (v < 0F) || (v > 1F) ) {
      v = 0.5F;
    }
    return v;
  }


  public static Image getLargestIconImage()
  {
    return largestIconImage;
  }


  public static float getNormalizedValue( JSlider slider )
  {
    float vNormalied = 0F;
    int   minValue   = slider.getMinimum();
    int   range      = slider.getMaximum() - minValue;
    if( range > 0 ) {
      int value  = slider.getValue();
      vNormalied = (float) (value - minValue) / (float) (range);
      if( vNormalied < 0F) {
	vNormalied = 0F;
      } else if( vNormalied > 1F ) {
	vNormalied = 1F;
      }
    }
    return vNormalied;
  }


  public static void getTableLayout(
				JTable     table,
				Properties props,
				String     propPrefix )
  {
    TableModel       tm  = table.getModel();
    TableColumnModel tcm = table.getColumnModel();
    if( (tm != null) && (tcm != null) ) {
      int n = tcm.getColumnCount();
      for( int i = 0; i < n; i++ ) {
	String colName  = null;
	int    colWidth = -1;
	TableColumn tc  = tcm.getColumn( i );
	if( tc != null ) {
	  colWidth     = tc.getWidth();
	  int modelIdx = tc.getModelIndex();
	  if( (modelIdx >= 0) && (modelIdx < tm.getColumnCount()) ) {
	    colName = tm.getColumnName( modelIdx );
	  }
	}
	props.setProperty(
		String.format( propPrefix + PROP_TABLE_COL_X_NAME, i ),
		colName != null ? colName : "" );
	props.setProperty(
		String.format( propPrefix + PROP_TABLE_COL_X_WIDTH, i ),
		String.valueOf( colWidth ) );
      }
      props.setProperty(
		propPrefix + PROP_TABLE_COL_COUNT,
		String.valueOf( n ) );
    }
  }


  public static Rectangle getViewRect( Component c )
  {
    Rectangle r = null;
    while( c != null ) {
      if( c instanceof JScrollPane ) {
	JViewport vp = ((JScrollPane) c).getViewport();
	if( vp != null ) {
	  r = vp.getViewRect();
	}
	break;
      }
      c = c.getParent();
    }
    return r != null ? r : c.getBounds();
  }


  public static void getWindowBounds(
				Window     window,
				Properties props,
				String     propPrefix )
  {
    getWindowBounds(
		window.getBounds(),
		isResizable( window ),
		props,
		propPrefix );
  }


  public static void getWindowBounds(
				Rectangle  windowBounds,
				boolean    resizable,
				Properties props,
				String     propPrefix )
  {
    if( windowBounds != null ) {
      props.setProperty(
		propPrefix + PROP_WINDOW_X,
		String.valueOf( windowBounds.x ) );
      props.setProperty(
		propPrefix + PROP_WINDOW_Y,
		String.valueOf( windowBounds.y ) );
      if( resizable ) {
	props.setProperty(
		propPrefix + PROP_WINDOW_WIDTH,
		String.valueOf( windowBounds.width ) );
	props.setProperty(
		propPrefix + PROP_WINDOW_HEIGHT,
		String.valueOf( windowBounds.height ) );
      }
    }
  }


  public static void restoreTableLayout(
				JTable     table,
				Properties props,
				String     propPrefix )
  {
    TableColumnModel tcm = table.getColumnModel();
    if( tcm != null ) {
      try {
	int nCols = getInt( props, propPrefix + PROP_TABLE_COL_COUNT );
	if( nCols > 0 ) {
	  boolean changed = moveTableColumns(
				tcm,
				props,
				propPrefix,
				nCols,
				true );
	  while( changed ) {
	    changed = moveTableColumns(
				tcm,
				props,
				propPrefix,
				nCols,
				false );
	  }
	}
      }
      catch( NullPointerException ex ) {}
      catch( NumberFormatException ex ) {}
    }
  }


  public static boolean restoreWindowBounds(
				Window     window,
				Properties props,
				String     propPrefix )
  {
    boolean rv = false;
    if( props != null ) {
      try {
	int x = getInt( props, propPrefix + PROP_WINDOW_X );
	int y = getInt( props, propPrefix + PROP_WINDOW_Y );
	int w = 0;
	int h = 0;
	if( isResizable( window ) ) {
	  w = getInt( props, propPrefix + PROP_WINDOW_WIDTH );
	  h = getInt( props, propPrefix + PROP_WINDOW_HEIGHT );
	  if( (w > 0) && (h > 0) ) {
	    window.setSize( w, h );
	  }
	} else {
	  w = window.getWidth();
	  h = window.getHeight();
	}

	// pruefen, ob die linke obere Fensterecke sichtbar ist
	boolean visible = false;
	w /= 4;
	h /= 4;
	for( GraphicsDevice gd : GraphicsEnvironment
					.getLocalGraphicsEnvironment()
					.getScreenDevices() )
	{
	  for( GraphicsConfiguration gc : gd.getConfigurations() ) {
	    Rectangle r = gc.getBounds();
	    if( r != null ) {
	      if( r.contains( x, y, w, h ) ) {
		visible = true;
		break;
	      }
	    }
	  }
	  if( visible ) {
	    break;
	  }
	}
	if( visible ) {
	  window.setLocation( x, y );
	  rv = true;
	}
      }
      catch( NullPointerException ex ) {}
      catch( NumberFormatException ex ) {}
    }
    return rv;
  }


  public static void setAudioPlayMixerName( String mixerName )
  {
    Main.setProperty(
		PROP_AUDIO_PLAY_MIXER,
		mixerName != null ? mixerName : "" );
  }


  public static void setAudioPlayVolume( float volume )
  {
    Main.setProperty( PROP_AUDIO_PLAY_VOLUME, String.valueOf( volume ) );
  }


  public static void setIconImagesAt( Window window )
  {
    if( iconImages == null ) {
      iconImages = new ArrayList<>();
      for( String resource : iconImageResources ) {
	try {
	  URL url = window.getClass().getResource( resource );
	  if( url != null ) {
	    Image image = window.getToolkit().createImage( url );
	    if( image != null ) {
	      iconImages.add( image );
	      largestIconImage = image;
	    }
	  }
	}
	catch( Exception ex ) {}
      }
    }
    if( !iconImages.isEmpty() ) {
      window.setIconImages( iconImages );
    }
  }


  public static void showErrorMsg(
				Component owner,
				String    msg,
				Exception ex )
  {
    String exMsg = null;
    if( ex != null ) {
      exMsg = ex.getMessage();
      if( exMsg != null ) {
	if( exMsg.isEmpty() ) {
	  exMsg = null;
	}
      }
      if( ex instanceof IOException ) {
	if( (msg == null) && (exMsg == null) ) {
	  msg = "Ein-/Ausgabefehler";
	}
      } else if( ex instanceof RuntimeException ) {
	if( exMsg == null ) {
	  exMsg = ex.getClass().getName();
	}
	if( msg == null ) {
	  msg = "Interner Programmfehler";
	}
      }
    }
    if( msg != null ) {
      if( exMsg != null ) {
	msg = msg + ":\n\n" + exMsg;
      }
    } else if( exMsg != null ) {
      msg = exMsg;
    }
    showErrorMsg( owner, msg );
  }


  public static void showErrorMsg( Component owner, String msg )
  {
    JOptionPane.showMessageDialog(
		owner,
		msg != null ? msg : "Unbekannter Fehler",
		"Fehler",
		JOptionPane.ERROR_MESSAGE );
  }


  public static File showOpenFileDlg(
			Component  owner,
			String     title,
			File       dirFile,
			FileFilter fileFilter )
  {
    JFileChooser fc = new JFileChooser();
    if( title != null ) {
      fc.setDialogTitle( title );
    }
    if( dirFile != null ) {
      if( !dirFile.isDirectory() ) {
	dirFile = dirFile.getParentFile();
      }
      if( dirFile != null ) {
	fc.setCurrentDirectory( dirFile );
      }
    }
    if( fileFilter != null ) {
      fc.addChoosableFileFilter( fileFilter );
      fc.setFileFilter( fileFilter );
    }
    return fc.showOpenDialog( owner ) == JFileChooser.APPROVE_OPTION ?
						fc.getSelectedFile()
						: null;
  }


  public static File showSaveFileDlg(
			Component        owner,
			String           title,
			File             dirFile,
			String           fileName,
			FileFilter       fileFilter,
			List<FileFilter> fileFilters )
  {
    File         rv = null;
    JFileChooser fc = new JFileChooser();
    if( title != null ) {
      fc.setDialogTitle( title );
    }
    if( dirFile != null ) {
      if( !dirFile.isDirectory() ) {
	dirFile = dirFile.getParentFile();
      }
      if( dirFile != null ) {
	fc.setCurrentDirectory( dirFile );
      }
    }
    if( fileName != null ) {
      if( dirFile != null ) {
	fc.setSelectedFile( new File( dirFile, fileName ) );
      } else {
	fc.setSelectedFile( new File( fileName ) );
      }
    }
    if( fileFilter != null ) {
      fc.addChoosableFileFilter( fileFilter );
      fc.setFileFilter( fileFilter );
    }
    if( fileFilters != null ) {
      if( !fileFilters.isEmpty() ) {
	for( FileFilter ff : fileFilters ) {
	  fc.addChoosableFileFilter( ff );
	}
	if( fileFilter == null ) {
	  fc.setFileFilter( fileFilters.get( 0 ) );
	}
      }
    }
    if( fc.showDialog( owner, "Speichern" ) == JFileChooser.APPROVE_OPTION ) {
      File file = fc.getSelectedFile();
      if( file != null ) {

	// ggf. fehlende Dateiendung auomatisch hinzufuegen
	FileFilter fFilter = fc.getFileFilter();
	String     fName   = file.getName();
	if( (fFilter != null) && (fName != null) ) {
	  if( (fFilter instanceof FileNameExtensionFilter)
	      && (fName.indexOf( '.' ) < 0) )
	  {
	    String[] e = ((FileNameExtensionFilter) fFilter).getExtensions();
	    if( e != null ) {
	      if( e.length == 1 ) {
		fName  = fName + '.' + e[ 0 ].toLowerCase();
		File d = file.getParentFile();
		if( d != null ) {
		  file = new File( d, fName );
		} else {
		  file = new File( fName );
		}
	      }
	    }
	  }
	}

	// Test auf Vorhandensein der Datei
	boolean status = true;
	if( file.exists() ) {
	  StringBuilder buf = new StringBuilder( 256 );
	  if( fName != null ) {
	    buf.append( fName );
	    buf.append( ":\n" );
	  }
	  buf.append( "Eine Datei mit dem Namen exisitiert bereits.\n"
		+ "M\u00F6chten Sie die Datei \u00FCberschreiben?" );
	  if( JOptionPane.showConfirmDialog(
				owner,
				buf.toString(),
				TITLE_CONFIRMATIOn,
				JOptionPane.YES_NO_OPTION,
				JOptionPane.WARNING_MESSAGE )
					== JOptionPane.YES_OPTION )
	  {
	    status = true;
	  }
	}
	if( status ) {
	  rv = file;
	}
      }
    }
    return rv;
  }


  public static void toFront( Frame frame )
  {
    frame.setVisible( true );
    int frameState = frame.getExtendedState();
    if( (frameState & Frame.ICONIFIED) != 0 ) {
      frame.setExtendedState( frameState & ~Frame.ICONIFIED );
    }
    frame.toFront();
  }


	/* --- private Methoden --- */

  private static int getInt( Properties props, String keyName )
  {
    if( props == null ) {
      throw new NullPointerException();
    }
    String value = props.getProperty( keyName );
    if( value == null ) {
      throw new NullPointerException();
    }
    return Integer.parseInt( value );
  }


  private static boolean isResizable( Window window )
  {
    boolean rv = false;
    if( window instanceof Dialog ) {
      rv = ((Dialog) window).isResizable();
    } if( window instanceof Frame ) {
      rv = ((Frame) window).isResizable();
    }
    return rv;
  }


  private static boolean moveTableColumns(
				TableColumnModel tcm,
				Properties       props,
				String           propPrefix,
				int              nCols,
				boolean          updWidths )
  {
    boolean changed = false;
    for( int i = 0; i < nCols; i++ ) {
      String colName = props.getProperty(
				String.format(
					propPrefix + PROP_TABLE_COL_X_NAME,
					i ) );
      if( colName != null ) {
	try {
	  int curIdx = tcm.getColumnIndex( colName );
	  if( curIdx >= 0 ) {
	    TableColumn tc = tcm.getColumn( curIdx );
	    if( tc != null ) {
	      if( updWidths ) {
		try {
		  int w = getInt(
				props,
				String.format(
					propPrefix + PROP_TABLE_COL_X_WIDTH,
					i ) );
		  if( w > 0 ) {
		    tc.setPreferredWidth( w );
		  }
		}
		catch( NullPointerException ex ) {}
		catch( NumberFormatException ex ) {}
	      }
	      if( curIdx != i ) {
		tcm.moveColumn( curIdx, i );
		changed = true;
	      }
	    }
	  }
	}
	catch( IllegalArgumentException ex ) {}
      }
    }
    return changed;
  }


	/* --- Konstruktor --- */

  private UIUtil()
  {
    // nicht instanziierbar
  }
}
