/*
 * (c) 2024-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Dialog zum Aendern der Einleseparameter 
 */

package jkcload.ui;

import java.awt.Dialog;
import java.awt.GridLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.Hashtable;
import java.util.Properties;
import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JSlider;
import javax.swing.SwingConstants;
import javax.swing.border.BevelBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import jkcload.Main;


public class LoadParamDlg extends JDialog
			implements ActionListener, ChangeListener
{
  private static final float DEFAULT_MIN_LEVEL    = 0.1F;
  private static final float DEFAULT_TOLERANCE    = 0.25F;
  private static final float DEFAULT_FLANK_STROKE = 0.15F;

  private static final String PROP_PREFIX     = Main.PROP_PREFIX + "load.";
  private static final String PROP_WIN_PREFIX = PROP_PREFIX + "param";
  private static final String PROP_MIN_LEVEL  = PROP_PREFIX + "level.min";
  private static final String PROP_TOLERANCE  = PROP_PREFIX + "tolerance";
  private static final String PROP_FLANK_STROKE
				= PROP_PREFIX + "flank_stroke";
  private static final String PROP_RECOGN_PHASES_BY
				= PROP_PREFIX + "recognize_phases_by";

  private static final String VALUE_DYNAMIC_AVG   = "dynamic_avg";
  private static final String VALUE_STEEP_FLANKES = "Steep_flankes";

  private JButton        btnOK;
  private JButton        btnCancel;
  private JButton        btnFlankStrokeMinus;
  private JButton        btnFlankStrokePlus;
  private JLabel         labelFlankStroke;
  private JSlider        sliderFlankStroke;
  private JSlider        sliderMinLevel;
  private JSlider        sliderTolerance;
  private JRadioButton   rbDynamicAvg;
  private JRadioButton   rbSteepFlanks;
  private WindowListener windowListener;


  public static float getFlankStrokeProperty()
  {
    Float v = getFloatProperty( PROP_FLANK_STROKE );
    return v != null ? v.floatValue() : DEFAULT_FLANK_STROKE;
  }


  public static float getMinLevelProperty()
  {
    Float v = getFloatProperty( PROP_MIN_LEVEL );
    return v != null ? v.floatValue() : DEFAULT_MIN_LEVEL;
  }


  public static boolean getSteepFlanksProperty()
  {
    String s = Main.getProperty( PROP_RECOGN_PHASES_BY );
    return s != null ? s.equals( VALUE_STEEP_FLANKES ) : false;
  }


  public static float getToleranceProperty()
  {
    Float v = getFloatProperty( PROP_TOLERANCE );
    return v != null ? v.floatValue() : DEFAULT_TOLERANCE;
  }


  public static void open( Window owner )
  {
    (new LoadParamDlg( owner )).setVisible( true );
  }


	/* --- ActionListener --- */

  @Override
  public void actionPerformed( ActionEvent e )
  {
    Object src = e.getSource();
    if( (src == this.rbDynamicAvg) || (src == this.rbSteepFlanks) ) {
      updFlankStrokeFieldsEnabled();
    } else if( src == this.btnFlankStrokeMinus ) {
      changeFlankStroke( -1 );
    } else if( src == this.btnFlankStrokePlus ) {
      changeFlankStroke( 1 );
    } else if( src == this.btnOK ) {
      doApply();
    } else if( src == this.btnCancel ) {
      doClose();
    }
  }


	/* --- ChangeListener --- */

  @Override
  public void stateChanged( ChangeEvent e )
  {
    if( e.getSource() == this.sliderFlankStroke )
      updFlankStrokeFieldsEnabled();
  }


	/* --- Konstruktor --- */

  private LoadParamDlg( Window owner )
  {
    super( owner, "Einleseparameter", Dialog.ModalityType.DOCUMENT_MODAL );
    setDefaultCloseOperation( JDialog.DO_NOTHING_ON_CLOSE );
    UIUtil.setIconImagesAt( this );


    // Fensterinhalt
    setLayout( new GridBagLayout() );

    GridBagConstraints gbc = new GridBagConstraints(
					0, 0,
					1, 1,
					0.0, 0.0,
					GridBagConstraints.WEST,
					GridBagConstraints.NONE,
					new Insets( 5, 5, 0, 5 ),
					0, 0 );

    add( new JLabel( "Mindestpegel:" ), gbc );

    Hashtable<Integer,JComponent> sliderLabels = new Hashtable<>();
    for( int percent = 0; percent <= 50; percent += 10 ) {
      sliderLabels.put(
		Integer.valueOf( percent ),
		new JLabel( String.format( "%d%%", percent ) ) );
    }

    this.sliderMinLevel = new JSlider(
				SwingConstants.HORIZONTAL,
				0,
				50,
				Math.round( DEFAULT_MIN_LEVEL * 100F ) );
    this.sliderMinLevel.setLabelTable( sliderLabels );
    this.sliderMinLevel.setMajorTickSpacing( 10 );
    this.sliderMinLevel.setMinorTickSpacing( 10 );
    this.sliderMinLevel.setPaintLabels( true );
    this.sliderMinLevel.setPaintTicks( true );
    this.sliderMinLevel.setPaintTrack( true );
    gbc.fill      = GridBagConstraints.HORIZONTAL;
    gbc.weightx   = 1.0;
    gbc.gridwidth = GridBagConstraints.REMAINDER;
    gbc.gridx++;
    add( this.sliderMinLevel, gbc );

    gbc.insets.top = 20;
    gbc.fill       = GridBagConstraints.NONE;
    gbc.weightx    = 0.0;
    gbc.gridwidth  = 1;
    gbc.gridx      = 0;
    gbc.gridy++;
    add( new JLabel( "Frequenztoleranz:" ), gbc );

    this.sliderTolerance = new JSlider(
				SwingConstants.HORIZONTAL,
				0,
				50,
				25 );
    this.sliderTolerance.setLabelTable( sliderLabels );
    this.sliderTolerance.setMajorTickSpacing( 10 );
    this.sliderTolerance.setMinorTickSpacing( 10 );
    this.sliderTolerance.setPaintLabels( true );
    this.sliderTolerance.setPaintTicks( true );
    this.sliderTolerance.setPaintTrack( true );
    gbc.fill      = GridBagConstraints.HORIZONTAL;
    gbc.weightx   = 1.0;
    gbc.gridwidth = GridBagConstraints.REMAINDER;
    gbc.gridx++;
    add( this.sliderTolerance, gbc );

    gbc.fill    = GridBagConstraints.NONE;
    gbc.weightx = 0.0;
    gbc.gridx   = 0;
    gbc.gridy++;
    add( new JLabel( "Algorithmus zur Erkennung der Phasenwechsel:" ), gbc );

    ButtonGroup grpPhaseChange = new ButtonGroup();

    this.rbDynamicAvg = new JRadioButton(
		"\u00DCber-/Unterschreiten des dynamischen Mittelwertes",
		true );
    grpPhaseChange.add( this.rbDynamicAvg );
    gbc.insets.top  = 0;
    gbc.insets.left = 50;
    gbc.gridy++;
    add( this.rbDynamicAvg, gbc );

    this.rbSteepFlanks = new JRadioButton( "Steile Flanken" );
    grpPhaseChange.add( this.rbSteepFlanks );
    gbc.gridy++;
    add( this.rbSteepFlanks, gbc );

    JPanel panelFlankStroke = new JPanel( new GridBagLayout() );
    panelFlankStroke.setBorder(
		BorderFactory.createBevelBorder( BevelBorder.LOWERED ) );
    gbc.insets.left = 100;
    gbc.fill        = GridBagConstraints.HORIZONTAL;
    gbc.weightx     = 1.0;
    gbc.gridy++;
    add( panelFlankStroke, gbc );

    GridBagConstraints gbcFlankStroke = new GridBagConstraints(
					0, 0,
					1, 1,
					0.0, 0.0,
					GridBagConstraints.WEST,
					GridBagConstraints.NONE,
					new Insets( 5, 5, 5, 5 ),
					0, 0 );

    this.labelFlankStroke = new JLabel( "Flankenh\u00F6he:" );
    panelFlankStroke.add( this.labelFlankStroke, gbcFlankStroke );

    this.btnFlankStrokeMinus = UIUtil.createImageButton(
						"/images/left.png",
						"<" );
    gbcFlankStroke.insets.left = 0;
    gbcFlankStroke.gridx++;
    panelFlankStroke.add( this.btnFlankStrokeMinus, gbcFlankStroke );

    sliderLabels = new Hashtable<>();
    for( int percent = 5; percent <= 45; percent += 10 ) {
      sliderLabels.put(
		Integer.valueOf( percent ),
		new JLabel( String.format( "%d%%", percent ) ) );
    }

    this.sliderFlankStroke = new JSlider(
				SwingConstants.HORIZONTAL,
				5,
				35,
				15 );
    this.sliderFlankStroke.setLabelTable( sliderLabels );
    this.sliderFlankStroke.setMajorTickSpacing( 10 );
    this.sliderFlankStroke.setMinorTickSpacing( 10 );
    this.sliderFlankStroke.setPaintLabels( true );
    this.sliderFlankStroke.setPaintTicks( true );
    this.sliderFlankStroke.setPaintTrack( true );
    gbcFlankStroke.fill    = GridBagConstraints.HORIZONTAL;
    gbcFlankStroke.weightx = 1.0;
    gbcFlankStroke.gridx++;
    panelFlankStroke.add( this.sliderFlankStroke, gbcFlankStroke );

    this.btnFlankStrokePlus = UIUtil.createImageButton(
					"/images/right.png",
					">" );
    gbcFlankStroke.fill    = GridBagConstraints.NONE;
    gbcFlankStroke.weightx = 0.0;
    gbcFlankStroke.gridx++;
    panelFlankStroke.add( this.btnFlankStrokePlus, gbcFlankStroke );

    JPanel panelBtn   = new JPanel( new GridLayout( 1, 2, 5, 5 ) );
    gbc.anchor        = GridBagConstraints.CENTER;
    gbc.insets.top    = 20;
    gbc.insets.left   = 5;
    gbc.insets.bottom = 10;
    gbc.fill          = GridBagConstraints.NONE;
    gbc.weightx       = 0.0;
    gbc.gridx         = 0;
    gbc.gridy++;
    add( panelBtn, gbc );

    this.btnOK = new JButton( "OK" );
    panelBtn.add( this.btnOK );

    this.btnCancel = new JButton( "Abbrechen" );
    panelBtn.add( this.btnCancel );


    // Voreinstellungen
    setSliderValue( this.sliderMinLevel, getMinLevelProperty() );
    setSliderValue( this.sliderTolerance, getToleranceProperty() );
    setSliderValue( this.sliderFlankStroke, getFlankStrokeProperty() );
    if( getSteepFlanksProperty() ) {
      this.rbSteepFlanks.setSelected( true );
    } else {
      this.rbDynamicAvg.setSelected( true );
    }
    updFlankStrokeFieldsEnabled();

    // Fenstergroesse und -position
    pack();
    setResizable( false );
    if( !UIUtil.restoreWindowBounds(
			this,
			Main.getProperties(),
			PROP_WIN_PREFIX ) )
    {
      setLocationByPlatform( true );
    }

    // Listener
    this.rbDynamicAvg.addActionListener( this );
    this.rbSteepFlanks.addActionListener( this );
    this.sliderFlankStroke.addChangeListener( this );
    this.btnFlankStrokeMinus.addActionListener( this );
    this.btnFlankStrokePlus.addActionListener( this );
    this.btnOK.addActionListener( this );
    this.btnCancel.addActionListener( this );
    this.windowListener = new WindowAdapter()
		{
		  @Override
		  public void windowClosing( WindowEvent e )
		  {
		    doClose();
		  }
		};
    addWindowListener( this.windowListener );
  }


	/* --- private Methoden --- */

  private void changeFlankStroke( int diffValue )
  {
    int value = this.sliderFlankStroke.getValue() + diffValue;
    if( (value >= this.sliderFlankStroke.getMinimum())
	&& (value <= this.sliderFlankStroke.getMaximum()) )
    {
      this.sliderFlankStroke.setValue( value );
      updFlankStrokeFieldsEnabled();
    }
  }


  private void doClose()
  {
    UIUtil.getWindowBounds(
			this,
			Main.getProperties(),
			PROP_WIN_PREFIX );
    this.rbDynamicAvg.removeActionListener( this );
    this.rbSteepFlanks.removeActionListener( this );
    this.sliderFlankStroke.removeChangeListener( this );
    this.btnFlankStrokeMinus.removeActionListener( this );
    this.btnFlankStrokePlus.removeActionListener( this );
    this.btnOK.removeActionListener( this );
    this.btnCancel.removeActionListener( this );
    removeWindowListener( this.windowListener );
    setVisible( false );
    dispose();
  }


  private void doApply()
  {
    Properties props = Main.getProperties();
    props.setProperty(
		PROP_MIN_LEVEL,
		String.valueOf(
			(float) this.sliderMinLevel.getValue() / 100F ) );
    props.setProperty(
		PROP_TOLERANCE,
		String.valueOf(
			(float) this.sliderTolerance.getValue() / 100F ) );
    props.setProperty(
		PROP_RECOGN_PHASES_BY,
		this.rbSteepFlanks.isSelected() ?
					VALUE_STEEP_FLANKES
					: VALUE_DYNAMIC_AVG );
    props.setProperty(
		PROP_FLANK_STROKE,
		String.valueOf(
			(float) this.sliderFlankStroke.getValue() / 100F ) );
    doClose();
  }


  private static Float getFloatProperty( String keyword )
  {
    Float  f = null;
    String s = Main.getProperty( keyword );
    if( s != null ) {
      try {
	float v = Float.parseFloat( s );
	if ( (v >= 0F) && (v <= 1F) ) {
	  f = Float.valueOf( v );
	}
      }
      catch( NumberFormatException ex ) {}
    }
    return f;
  }


  private static void setSliderValue( JSlider slider, float normalizedValue )
  {
    int v = Math.round( normalizedValue * 100F );
    if( v < slider.getMinimum() ) {
      v = slider.getMinimum();
    } else if( v > slider.getMaximum() ) {
      v = slider.getMaximum();
    }
    slider.setValue( v );
  }


  private void updFlankStrokeFieldsEnabled()
  {
    boolean state = this.rbSteepFlanks.isSelected();
    this.labelFlankStroke.setEnabled( state );
    this.btnFlankStrokeMinus.setEnabled( state );
    this.btnFlankStrokePlus.setEnabled( state );
    this.sliderFlankStroke.setEnabled( state );
    if( state ) {
      int value = this.sliderFlankStroke.getValue();
      if( value <= this.sliderFlankStroke.getMinimum() ) {
	this.btnFlankStrokeMinus.setEnabled( false );
      }
      if( value >= this.sliderFlankStroke.getMaximum() ) {
	this.btnFlankStrokePlus.setEnabled( false );
      }
    }
  }
}
