/*
 * (c) 2023-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Applikationsfenster
 */

package jkcload.ui;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.Font;
import java.awt.Image;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.dnd.DnDConstants;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetEvent;
import java.awt.dnd.DropTargetListener;
import java.awt.dnd.InvalidDnDOperationException;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.Mixer;
import javax.sound.sampled.UnsupportedAudioFileException;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.JSlider;
import javax.swing.JTable;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.swing.table.TableColumnModel;
import jkcload.Main;
import jkcload.audio.AudioFileThread;
import jkcload.audio.AudioLineThread;
import jkcload.audio.AudioThread;
import jkcload.audio.AudioUtil;
import jkcload.audio.RecognitionSettings;


public class LoadFrm extends JFrame implements
					ActionListener,
					ChangeListener,
					DropTargetListener,
					ListSelectionListener,
					MouseListener,
					AudioThread.Observer
{
  private static final String ITEM_AUDIO_OPEN_LINE
					= "Audiokanal \u00F6ffnen...";
  private static final String ITEM_AUDIO_OPEN_FILE
					= "Audiodatei \u00F6ffnen...";
  private static final String ITEM_AUDIO_REOPEN_FILE
					= "erneut \u00F6ffnen";
  private static final String ITEM_AUDIO_CLOSE
					= "Audioquelle schlie\u00Dfen";
  private static final String ITEM_FILE_REMOVE     = "Entfernen";
  private static final String ITEM_FILE_REMOVE_ALL = "Alle entfernen";
  private static final String ITEM_FILE_REMOVE_ERR
					= "Alle fehlerhaften entfernen";
  private static final String ITEM_FILE_SAVE_AS = "Speichern unter...";
  private static final String ITEM_FILE_SHOW_CONTENT
						= "Dateiinhalt anzeigen...";
  private static final String ITEM_FILE_SHOW_DIFF
			= "Dateiinhalte und Unterschiede anzeigen...";
  private static final String ITEM_FILE_SHOW_LOG  = "Protokoll anzeigen...";
  private static final String LABEL_PROGRES_DEFAULT = "Pegel/Fortschritt:";
  private static final String TEXT_MONITOR          = "Mith\u00F6ren";

  private static final String PROP_VERSION = Main.PROP_PREFIX + "version";
  private static final String PROP_PREFIX  = Main.PROP_PREFIX + "load.";
  private static final String PROP_FORMAT  = PROP_PREFIX + "format";
  private static final String PROP_MONITOR_ENABLED
				= PROP_PREFIX + "monitor.enabled";
  private static final String PROP_ANALYSIS_ENABLED
				= Main.PROP_PREFIX + "analysis.enabled";
  private static final String PROP_AUDIO_REC_MIXER
				= Main.PROP_PREFIX + "audio.rec.mixer.name";


  private static final int VOLUME_SLIDER_MAX = 1000;

  private JButton            btnAudioOpenFile;
  private JButton            btnAudioReopen;
  private JButton            btnAudioClose;
  private JButton            btnLoadParams;
  private JPopupMenu         popupFile;
  private JMenuItem          popupFileShowContent;
  private JMenuItem          popupFileShowLog;
  private JMenuItem          popupFileSaveAs;
  private JMenuItem          popupFileRemove;
  private JMenuItem          popupFileRemoveErr;
  private JMenuItem          popupFileRemoveAll;
  private JMenuItem          mnuAbout;
  private JCheckBoxMenuItem  mnuAnalysisRec;
  private JMenuItem          mnuAnalysisShow;
  private JMenuItem          mnuAudioFileConcat;
  private JMenuItem          mnuAudioOpenLine;
  private JMenuItem          mnuAudioOpenFile;
  private JMenuItem          mnuAudioClose;
  private JMenuItem          mnuFileFindDoublets;
  private JMenuItem          mnuFileShowContent;
  private JMenuItem          mnuFileShowLog;
  private JMenuItem          mnuFileSaveAs;
  private JMenuItem          mnuFileRemove;
  private JMenuItem          mnuFileRemoveErr;
  private JMenuItem          mnuFileRemoveAll;
  private JMenuItem          mnuQuit;
  private JMenuItem          mnuHelpHome;
  private JMenuItem          mnuLicense;
  private JMenuItem          mnuSettingsAudio;
  private JMenuItem          mnuSettingsLoadParams;
  private JMenuItem          mnuSettingsBC;
  private JMenuItem          mnuSettingsDelete;
  private JMenuItem          mnuSettingsSave;
  private JTextField         fldAudioSrc;
  private JTextField         fldAudioInfo;
  private JTextField         fldReadActivity;
  private JLabel             labelAudioInfo;
  private JLabel             labelProgress;
  private JLabel             labelReadActivity;
  private JLabel             labelRecFmt;
  private JProgressBar       progressBar;
  private VolumeBar          volumeBar;
  private JComboBox<Object>  comboRecFmt;
  private JCheckBox          cbMonitor;
  private JSlider            sliderMonitorVolume;
  private JScrollPane        fileScrollPane;
  private JTable             fileTable;
  private ListSelectionModel fileSelectionModel;
  private FileTableModel     fileTableModel;
  private DropTarget         dropTarget1;
  private DropTarget         dropTarget2;
  private DropTarget         dropTarget3;
  private AudioThread        audioThread;
  private AnalysisFrm        analysisFrm;
  private AudioFileConcatFrm audioFileConcatFrm;
  private HelpFrm            helpFrm;
  private HexAsciiDiffFrm    hexAsciiDiffFrm;
  private AudioFormat        recentAudioFmt;
  private Map<String,String> recentFormat2FileExt;
  private File               recentSaveDir;
  private File               recentAudioDir;
  private Object             recentAudioSrc;
  private byte[]             recentAudioData;
  private byte[]             recentPhaseData;
  private int                recentChannel;
  private boolean            suppressSettingsSavedMsg;

  public LoadFrm()
  {
    this.recentChannel            = 0;
    this.recentSaveDir            = null;
    this.recentAudioDir           = null;
    this.recentAudioSrc           = null;
    this.recentAudioFmt           = null;
    this.recentAudioData          = null;
    this.recentPhaseData          = null;
    this.recentFormat2FileExt     = new HashMap<>();
    this.suppressSettingsSavedMsg = false;
    this.analysisFrm              = null;
    this.audioFileConcatFrm       = null;
    this.helpFrm                  = null;
    this.hexAsciiDiffFrm          = null;
    setDefaultCloseOperation( JFrame.DO_NOTHING_ON_CLOSE );
    setTitle( Main.APPNAME );


    // Fenster-Icon
    UIUtil.setIconImagesAt( this );


    // Menu Datei
    JMenu mnuFile = new JMenu( "Datei" );
    mnuFile.setMnemonic( KeyEvent.VK_D );

    this.mnuAudioOpenLine = new JMenuItem( ITEM_AUDIO_OPEN_LINE );
    mnuFile.add( this.mnuAudioOpenLine );

    this.mnuAudioOpenFile = new JMenuItem( ITEM_AUDIO_OPEN_FILE );
    mnuFile.add( this.mnuAudioOpenFile );

    this.mnuAudioClose = new JMenuItem( ITEM_AUDIO_CLOSE );
    mnuFile.add( this.mnuAudioClose );
    mnuFile.addSeparator();

    this.mnuFileShowContent = new JMenuItem( ITEM_FILE_SHOW_DIFF );
    mnuFile.add( this.mnuFileShowContent );

    this.mnuFileShowLog = new JMenuItem( ITEM_FILE_SHOW_LOG );
    mnuFile.add( this.mnuFileShowLog );

    this.mnuFileFindDoublets = new JMenuItem( "Dubletten finden" );
    mnuFile.add( this.mnuFileFindDoublets );
    mnuFile.addSeparator();

    this.mnuFileRemove = new JMenuItem( ITEM_FILE_REMOVE );
    this.mnuFileRemove.setAccelerator(
		KeyStroke.getKeyStroke( KeyEvent.VK_DELETE, 0 ) );
    mnuFile.add( this.mnuFileRemove );

    this.mnuFileRemoveErr = new JMenuItem( ITEM_FILE_REMOVE_ERR );
    mnuFile.add( this.mnuFileRemoveErr );

    this.mnuFileRemoveAll = new JMenuItem( ITEM_FILE_REMOVE_ALL );
    mnuFile.add( this.mnuFileRemoveAll );
    mnuFile.addSeparator();

    this.mnuFileSaveAs = new JMenuItem( ITEM_FILE_SAVE_AS );
    mnuFile.add( this.mnuFileSaveAs );
    mnuFile.addSeparator();

    this.mnuQuit = new JMenuItem( "Beenden" );
    mnuFile.add( this.mnuQuit );


    // Menu Analyse
    JMenu mnuAnalysis = new JMenu( "Analyse" );
    mnuAnalysis.setMnemonic( KeyEvent.VK_A );

    this.mnuAnalysisRec = new JCheckBoxMenuItem(
					"Analysedaten aufzeichnen" );
    mnuAnalysis.add( this.mnuAnalysisRec );

    this.mnuAnalysisShow = new JMenuItem( "Analysedaten anzeigen..." );
    mnuAnalysis.add( this.mnuAnalysisShow );


    // Menu Extra
    JMenu mnuExtra = new JMenu( "Extra" );
    mnuExtra.setMnemonic( KeyEvent.VK_X );

    this.mnuAudioFileConcat = new JMenuItem(
					"Audiodateien zusammenf\u00FCgen" );
    mnuExtra.add( this.mnuAudioFileConcat );


    // Menu Einstellungen
    JMenu mnuSettings = new JMenu( "Einstellungen" );
    mnuSettings.setMnemonic( KeyEvent.VK_E );

    this.mnuSettingsAudio = new JMenuItem(
				"Audiowiedergabeger\u00E4t festlegen..." );
    mnuSettings.add( this.mnuSettingsAudio );

    this.mnuSettingsLoadParams = new JMenuItem(
			"Parameter f\u00FCr das Einlesen..." );
    mnuSettings.add( this.mnuSettingsLoadParams );

    this.mnuSettingsBC = new JMenuItem( "BASICODE-Einstellungen..." );
    mnuSettings.add( this.mnuSettingsBC );

    File settingsFile = Main.getSettingsFile();
    if( settingsFile != null ) {
      mnuSettings.addSeparator();

      this.mnuSettingsSave = new JMenuItem( "Einstellungen speichern" );
      mnuSettings.add( this.mnuSettingsSave );

      this.mnuSettingsDelete = new JMenuItem(
				"Gespeicherte Einstellungen l\u00F6schen" );
      this.mnuSettingsDelete.setEnabled( settingsFile.exists() );
      mnuSettings.add( this.mnuSettingsDelete );
    } else {
      this.mnuSettingsSave   = null;
      this.mnuSettingsDelete = null;
    }


    // Menu Hilfe
    JMenu mnuHelp = new JMenu( "Hilfe" );
    mnuHelp.setMnemonic( KeyEvent.VK_H );

    this.mnuHelpHome = new JMenuItem( "Hilfe..." );
    mnuHelp.add( this.mnuHelpHome );
    mnuHelp.addSeparator();

    this.mnuAbout = new JMenuItem( "\u00DCber " +  Main.APPNAME + "..." );
    mnuHelp.add( this.mnuAbout );

    this.mnuLicense = new JMenuItem( "Lizenzbestimmungen..." );
    mnuHelp.add( this.mnuLicense );


    // Menuleiste
    JMenuBar mnuBar = new JMenuBar();
    mnuBar.add( mnuFile );
    mnuBar.add( mnuAnalysis );
    mnuBar.add( mnuExtra );
    mnuBar.add( mnuSettings );
    mnuBar.add( mnuHelp );
    setJMenuBar( mnuBar );


    // Popup-Menu Datei
    this.popupFile = new JPopupMenu();

    this.popupFileShowContent = new JMenuItem( ITEM_FILE_SHOW_DIFF );
    this.popupFile.add( this.popupFileShowContent );

    this.popupFileShowLog = new JMenuItem( ITEM_FILE_SHOW_LOG );
    this.popupFile.add( this.popupFileShowLog );
    this.popupFile.addSeparator();

    this.popupFileRemove = new JMenuItem( ITEM_FILE_REMOVE );
    this.popupFile.add( this.popupFileRemove );

    this.popupFileRemoveErr = new JMenuItem( ITEM_FILE_REMOVE_ERR );
    this.popupFile.add( this.popupFileRemoveErr );

    this.popupFileRemoveAll = new JMenuItem( ITEM_FILE_REMOVE_ALL );
    this.popupFile.add( this.popupFileRemoveAll );
    this.popupFile.addSeparator();

    this.popupFileSaveAs = new JMenuItem( ITEM_FILE_SAVE_AS );
    this.popupFile.add( this.popupFileSaveAs );


    // Fensterinhalt
    setLayout( new GridBagLayout() );

    GridBagConstraints gbc = new GridBagConstraints(
					0, 0,
					1, 1,
					0.0, 0.0,
					GridBagConstraints.EAST,
					GridBagConstraints.NONE,
					new Insets( 5, 5, 0, 5 ),
					0, 0 );

    this.labelRecFmt = new JLabel( "Aufnahmeformat:" );
    add( this.labelRecFmt, gbc );

    gbc.gridy++;
    add( new JLabel( "Audioquelle:" ), gbc );

    this.labelAudioInfo = new JLabel( "Info:" );
    gbc.gridy++;
    add( this.labelAudioInfo, gbc );

    this.labelProgress = new JLabel( LABEL_PROGRES_DEFAULT );
    gbc.gridy++;
    add( this.labelProgress, gbc );

    this.labelReadActivity = new JLabel( "Aktivit\u00E4t:" );
    gbc.gridy++;
    add( this.labelReadActivity, gbc );

    int gridyTable = gbc.gridy + 1;

    JPanel panelRecSettings = new JPanel();
    gbc.anchor              = GridBagConstraints.WEST;
    gbc.gridy               = 0;
    gbc.gridx++;
    add( panelRecSettings, gbc );

    panelRecSettings.setLayout(
		new BoxLayout( panelRecSettings, BoxLayout.X_AXIS ) );

    this.comboRecFmt = new JComboBox<>();
    this.comboRecFmt.addItem( "--- Bitte ausw\u00E4hlen ---" );
    for( RecognitionSettings.Format fmt
			: RecognitionSettings.Format.values() )
    {
      this.comboRecFmt.addItem( fmt );
    }
    panelRecSettings.add( this.comboRecFmt );
    panelRecSettings.add( Box.createHorizontalStrut( 5 ) );

    this.btnLoadParams = new JButton( "Parameter..." );
    panelRecSettings.add( this.btnLoadParams );

    JPanel panelAudioSrc = new JPanel( new GridBagLayout() );
    gbc.fill             = GridBagConstraints.HORIZONTAL;
    gbc.weightx          = 1.0;
    gbc.gridy++;
    add( panelAudioSrc, gbc );

    GridBagConstraints gbcAudioSrc = new GridBagConstraints(
					0, 0,
					1, 1,
					1.0, 0.0,
					GridBagConstraints.WEST,
					GridBagConstraints.HORIZONTAL,
					new Insets( 0, 0, 0, 0 ),
					0, 0 );

    this.fldAudioSrc = new JTextField();
    this.fldAudioSrc.setEditable( false );
    panelAudioSrc.add( this.fldAudioSrc, gbcAudioSrc );

    this.btnAudioOpenFile = UIUtil.createImageButton(
						"/images/open.png",
						"Datei \u00F6ffnen..." );
    gbcAudioSrc.fill        = GridBagConstraints.NONE;
    gbcAudioSrc.weightx     = 0.0;
    gbcAudioSrc.insets.left = 5;
    gbcAudioSrc.gridx++;
    panelAudioSrc.add( this.btnAudioOpenFile, gbcAudioSrc );

    this.btnAudioReopen = UIUtil.createImageButton(
			"/images/reopen.png",
			"Audioquelle erneut \u00F6ffnen und einlesen" );
    this.btnAudioReopen.setEnabled( false );
    gbcAudioSrc.gridx++;
    panelAudioSrc.add( this.btnAudioReopen, gbcAudioSrc );

    this.btnAudioClose = UIUtil.createImageButton(
				"/images/stop.png",
				"Stopp, Audioquelle schlie\u00DFen" );
    gbcAudioSrc.gridx++;
    panelAudioSrc.add( this.btnAudioClose, gbcAudioSrc );

    this.fldAudioInfo = new JTextField();
    this.fldAudioInfo.setEditable( false );
    gbc.gridy++;
    add( this.fldAudioInfo, gbc );

    JPanel panelProgress = new JPanel( new GridBagLayout() );
    gbc.gridy++;
    add( panelProgress, gbc );

    GridBagConstraints gbcProgress = new GridBagConstraints(
					0, 0,
					1, 1,
					1.0, 0.0,
					GridBagConstraints.WEST,
					GridBagConstraints.HORIZONTAL,
					new Insets( 0, 0, 0, 0 ),
					0, 0 );

    this.progressBar = new JProgressBar( SwingConstants.HORIZONTAL );
    this.progressBar.setBorderPainted( true );
    panelProgress.add( this.progressBar, gbcProgress );
    this.volumeBar = new VolumeBar( this.progressBar );

    URL urlSpeakerOn  = getClass().getResource( "/images/speaker_on.png" );
    URL urlSpeakerOff = getClass().getResource( "/images/speaker_off.png" );
    if( (urlSpeakerOn != null) && (urlSpeakerOff != null) ) {
      this.cbMonitor = new JCheckBox( new ImageIcon( urlSpeakerOff ) );
      this.cbMonitor.setSelectedIcon( new ImageIcon( urlSpeakerOn ) );
      this.cbMonitor.setToolTipText( TEXT_MONITOR );
    } else {
      this.cbMonitor = new JCheckBox( TEXT_MONITOR );
    }
    gbcProgress.fill        = GridBagConstraints.NONE;
    gbcProgress.weightx     = 0.0;
    gbcProgress.insets.left = 20;
    gbcProgress.gridx++;
    panelProgress.add( this.cbMonitor, gbcProgress );

    float volume = UIUtil.getAudioPlayVolume();

    this.sliderMonitorVolume = new JSlider(
		SwingConstants.HORIZONTAL,
		0,
		VOLUME_SLIDER_MAX,
		Math.round( volume * (float) VOLUME_SLIDER_MAX ) );
    this.sliderMonitorVolume.setPaintTicks( true );
    this.sliderMonitorVolume.setPaintTrack( true );
    this.sliderMonitorVolume.setPaintLabels( false );
    this.sliderMonitorVolume.setToolTipText( "Lautst\u00E4rke" );

    Dimension prefSize = this.sliderMonitorVolume.getPreferredSize();
    if( prefSize != null ) {
      this.sliderMonitorVolume.setPreferredSize(
		new Dimension( 100, prefSize.height ) );
    }
    gbcProgress.insets.left = 5;
    gbcProgress.gridx++;
    panelProgress.add( this.sliderMonitorVolume, gbcProgress );

    this.fldReadActivity = new JTextField();
    this.fldReadActivity.setEditable( false );
    gbc.gridy++;
    add( this.fldReadActivity, gbc );

    this.fileTableModel = new FileTableModel();
    this.fileTable      = new JTable( this.fileTableModel );
    this.fileTable.setAutoResizeMode( JTable.AUTO_RESIZE_OFF );
    this.fileTable.setCellSelectionEnabled( false );
    this.fileTable.setColumnSelectionAllowed( false );
    this.fileTable.setRowSelectionAllowed( true );
    this.fileTable.setDragEnabled( false );
    this.fileTable.setFillsViewportHeight( true );
    this.fileTable.setDefaultRenderer(
			String.class,
			new FileTableCellRenderer() );
    this.fileTable.setSelectionMode(
			ListSelectionModel.MULTIPLE_INTERVAL_SELECTION );
    this.fileTable.setShowGrid( false );
    fileScrollPane    = new JScrollPane( this.fileTable );
    gbc.insets.top    = 20;
    gbc.insets.bottom = 5;
    gbc.fill          = GridBagConstraints.BOTH;
    gbc.weightx       = 1.0;
    gbc.weighty       = 1.0;
    gbc.gridwidth     = GridBagConstraints.REMAINDER;
    gbc.gridx         = 0;
    gbc.gridy         = gridyTable;
    add( fileScrollPane, gbc );

    int[] colWidths = { 150, 60, 60, 60, 150, 200, 60, 100, 100 };
    int   wTable    = 0;
    for( int w : colWidths ) {
      wTable += w;
    }
    this.fileTable.setPreferredScrollableViewportSize(
				new Dimension( wTable, 300 ) );

    TableColumnModel tcm = this.fileTable.getColumnModel();
    if( tcm != null ) {
      int n = Math.min( colWidths.length, tcm.getColumnCount() );
      for( int i = 0; i < n; i++ ) {
	tcm.getColumn( i ).setPreferredWidth( colWidths[ i ] );
      }
    }

    this.fileSelectionModel = this.fileTable.getSelectionModel();
    if( this.fileSelectionModel != null ) {
      this.mnuFileRemove.setEnabled( false );
      this.mnuFileSaveAs.setEnabled( false );
      this.mnuFileShowContent.setEnabled( false );
      this.mnuFileShowLog.setEnabled( false );
      this.popupFileRemove.setEnabled( false );
      this.popupFileSaveAs.setEnabled( false );
      this.popupFileShowContent.setEnabled( false );
      this.popupFileShowLog.setEnabled( false );
    }


    // Fenstergroesse, -position und Voreinstellung
    Properties props = Main.getProperties();
    UIUtil.restoreTableLayout( this.fileTable, props, PROP_PREFIX );
    if( !UIUtil.restoreWindowBounds( this, props, PROP_PREFIX ) ) {
      pack();
      setLocationByPlatform( true );
    }
    if( props != null ) {
      String analysisEnabled = props.getProperty( PROP_ANALYSIS_ENABLED );
      if( analysisEnabled != null ) {
	this.mnuAnalysisRec.setSelected(
			Boolean.parseBoolean( analysisEnabled ) );
      }
      String monitorEnabled = props.getProperty( PROP_MONITOR_ENABLED );
      if( monitorEnabled != null ) {
	this.cbMonitor.setSelected(
			Boolean.parseBoolean( monitorEnabled ) );
      }
      String format = props.getProperty( PROP_FORMAT );
      if( format != null ) {
	try {
	  this.comboRecFmt.setSelectedItem(
		RecognitionSettings.Format.valueOf( format ) );
	}
	catch( IllegalArgumentException ex ) {}
      }
    }
    updFileActionsEnabled();
    setAudioStatus( false );


    // Drag&Drop
    this.dropTarget1 = new DropTarget( this.fldAudioSrc, this );
    this.dropTarget2 = new DropTarget( this.fileTable, this );
    this.dropTarget3 = new DropTarget( this.fileScrollPane, this );
    this.dropTarget1.setActive( true );
    this.dropTarget2.setActive( true );
    this.dropTarget3.setActive( true );


    // Listener
    this.btnAudioOpenFile.addActionListener( this );
    this.btnAudioReopen.addActionListener( this );
    this.btnAudioClose.addActionListener( this );
    this.mnuAbout.addActionListener( this );
    this.mnuAnalysisShow.addActionListener( this );
    this.mnuAudioFileConcat.addActionListener( this );
    this.mnuAudioOpenLine.addActionListener( this );
    this.mnuAudioOpenFile.addActionListener( this );
    this.mnuAudioClose.addActionListener( this );
    this.mnuFileFindDoublets.addActionListener( this );
    this.mnuFileRemove.addActionListener( this );
    this.mnuFileRemoveErr.addActionListener( this );
    this.mnuFileRemoveAll.addActionListener( this );
    this.mnuFileSaveAs.addActionListener( this );
    this.mnuFileShowContent.addActionListener( this );
    this.mnuFileShowLog.addActionListener( this );
    this.mnuHelpHome.addActionListener( this );
    this.mnuLicense.addActionListener( this );
    this.mnuQuit.addActionListener( this );
    this.mnuSettingsAudio.addActionListener( this );
    this.mnuSettingsLoadParams.addActionListener( this );
    this.mnuSettingsBC.addActionListener( this );
    if( this.mnuSettingsDelete != null ) {
      this.mnuSettingsDelete.addActionListener( this );
    }
    if( this.mnuSettingsSave != null ) {
      this.mnuSettingsSave.addActionListener( this );
    }
    this.popupFileRemove.addActionListener( this );
    this.popupFileRemoveErr.addActionListener( this );
    this.popupFileRemoveAll.addActionListener( this );
    this.popupFileSaveAs.addActionListener( this );
    this.popupFileShowContent.addActionListener( this );
    this.popupFileShowLog.addActionListener( this );
    this.cbMonitor.addActionListener( this );
    this.sliderMonitorVolume.addChangeListener( this );
    this.btnLoadParams.addActionListener( this );
    this.fileTable.addMouseListener( this );
    this.fileScrollPane.addMouseListener( this );
    if( this.fileSelectionModel != null ) {
      this.fileSelectionModel.addListSelectionListener( this );
    }
    addWindowListener(
		new WindowAdapter()
		{
		  @Override
		  public void windowClosing( WindowEvent e )
		  {
		    doQuit();
		  }
		} );
  }


	/* --- ActionListener --- */

  @Override
  public void actionPerformed( ActionEvent e )
  {
    Object src = e.getSource();
    if( src == this.mnuAbout ) {
      doAbout();
    } else if( src == this.mnuHelpHome ) {
      doHelp( HelpFrm.HOME_PAGE );
    } else if( src == this.mnuLicense ) {
      doHelp( HelpFrm.LICENSE_PAGE );
    } else if( src == this.mnuAnalysisShow ) {
      doAnalysisShow();
    } else if( src == this.mnuAudioFileConcat ) {
      doAudioFileConcat();
    } else if( (src == this.mnuAudioOpenFile)
	       || (src == this.btnAudioOpenFile) )
    {
      doAudioOpenFile();
    } else if( src == this.mnuAudioOpenLine ) {
      doAudioOpenLine();
    } else if( (src == this.mnuAudioClose)
	       || (src == this.btnAudioClose) )
    {
      doAudioClose();
    } else if( (src == this.mnuFileShowContent)
	       || (src == this.popupFileShowContent) )
    {
      doFileShowContent();
    } else if( src == this.mnuFileFindDoublets ) {
      doFileFindDoublets();
    } else if( (src == this.mnuFileShowLog)
	       || (src == this.popupFileShowLog) )
    {
      doFileShowLog();
    } else if( (src == this.mnuFileRemove)
	       || (src == this.popupFileRemove) )
    {
      doFileRemove();
    } else if( (src == this.mnuFileRemoveErr)
	       || (src == this.popupFileRemoveErr) )
    {
      doFileRemoveErr();
    } else if( (src == this.mnuFileRemoveAll)
	       || (src == this.popupFileRemoveAll) )
    {
      doFileRemoveAll();
    } else if( (src == this.mnuFileSaveAs)
	       || (src == this.popupFileSaveAs) )
    {
      doFileSaveAs();
    } else if( src == this.mnuQuit ) {
      doQuit();
    } else if( src == this.mnuSettingsAudio ) {
      doSettingsAudio();
    } else if( (src == this.mnuSettingsLoadParams)
	       || (src == this.btnLoadParams) )
    {
      LoadParamDlg.open( this );
    } else if( src == this.mnuSettingsBC ) {
      BCSettingsDlg.open( this );
    } else if( src == this.mnuSettingsDelete ) {
      doSettingsDelete();
    } else if( src == this.mnuSettingsSave ) {
      doSettingsSave();
    } else if( src == this.btnAudioReopen ) {
      doAudioReopen();
    } else if( src == this.cbMonitor ) {
      requestedMonitorStateChanged();
    }
  }


	/* --- ChangeListener --- */

  @Override
  public void stateChanged( ChangeEvent e )
  {
    if( e.getSource() == this.sliderMonitorVolume ) {
      float       volume      = getNormalizedMonitorVolume();
      AudioThread audioThread = this.audioThread;
      if( audioThread != null ) {
	audioThread.setMonitorVolume( volume );
      }
      UIUtil.setAudioPlayVolume( volume );
    }
  }


	/* --- DropTargetListener --- */

  @Override
  public void dragEnter( DropTargetDragEvent e )
  {
    UIUtil.acceptDragInCaseOfFileTransfer( e );
  }


  @Override
  public void dragExit( DropTargetEvent e )
  {
    // leer
  }


  @Override
  public void dragOver( DropTargetDragEvent e )
  {
    UIUtil.acceptDragInCaseOfFileTransfer( e );
  }


  @Override
  public void drop( DropTargetDropEvent e )
  {
    RecognitionSettings recSettings = checkAndGetRecognitionSettings();
    if( (recSettings != null) && (this.audioThread == null) ) {
      e.acceptDrop( DnDConstants.ACTION_COPY );
      try {
	Transferable t = e.getTransferable();
	if( t != null ) {
	  if( t.isDataFlavorSupported( DataFlavor.javaFileListFlavor ) ) {
	    Object c = t.getTransferData( DataFlavor.javaFileListFlavor );
	    if( c != null ) {
	      if( c instanceof Collection ) {
		Iterator iter = ((Collection) c).iterator();
		if( iter.hasNext() ) {
		  Object f = iter.next();
		  if( f != null ) {
		    if( f instanceof File ) {
		      EventQueue.invokeLater( ()->openAudioFile(
							recSettings,
							(File) f,
							-1 ) );
		    }
		  }
		}
	      }
	    }
	  }
	}
      }
      catch( Exception ex ) {
	UIUtil.fireShowErrorDnDOperationFailed( this, ex );
      }
    }
  }


  @Override
  public void dropActionChanged( DropTargetDragEvent e )
  {
    // leer
  }


	/* --- ListSelectionListener --- */

  @Override
  public void valueChanged( ListSelectionEvent e )
  {
    if( e.getSource() == this.fileSelectionModel ) {
      int     n        = this.fileTable.getSelectedRowCount();
      boolean state    = (n > 0);
      boolean stateOne = (n == 1);
      boolean stateLog = false;
      if( stateOne ) {
	FileData fileData = getFileDataByViewRowIdx(
					this.fileTable.getSelectedRow() );
	if( fileData != null ) {
	  stateLog = fileData.hasLog();
	}
      }
      this.mnuFileRemove.setEnabled( state );
      this.mnuFileSaveAs.setEnabled( state );
      this.mnuFileShowContent.setText(
		n == 1 ? ITEM_FILE_SHOW_CONTENT : ITEM_FILE_SHOW_DIFF );
      this.mnuFileShowContent.setEnabled( state );
      this.mnuFileShowLog.setEnabled( stateLog );
      this.popupFileRemove.setEnabled( state );
      this.popupFileSaveAs.setEnabled( state );
      this.popupFileShowContent.setText(
		n == 1 ? ITEM_FILE_SHOW_CONTENT : ITEM_FILE_SHOW_DIFF );
      this.popupFileShowContent.setEnabled( state );
      this.popupFileShowLog.setEnabled( stateLog );
    }
  }


	/* --- MouseListener --- */

  @Override
  public void mouseClicked( MouseEvent e )
  {
    if( e.isPopupTrigger() )
      popupTriggered( e );
  }


  @Override
  public void mouseEntered( MouseEvent e )
  {
    // leer
  }


  @Override
  public void mouseExited( MouseEvent e )
  {
    // leer
  }


  @Override
  public void mousePressed( MouseEvent e )
  {
    if( e.isPopupTrigger() )
      popupTriggered( e );
  }


  @Override
  public void mouseReleased( MouseEvent e )
  {
    if( e.isPopupTrigger() )
      popupTriggered( e );
  }


	/* --- AudioThread.Observer --- */

  @Override
  public void analysisData(
			AudioFormat audioFmt,
			byte[]      audioData,
			byte[]      phaseData )
  {
    this.recentAudioFmt  = audioFmt;
    this.recentAudioData = audioData;
    this.recentPhaseData = phaseData;
    fireShowAnalysisData();
  }


  @Override
  public void audioStatusChanged(
				AudioThread audioThread,
				boolean     active,
				String      formatInfo,
				String      errMsg )
  {
    EventQueue.invokeLater( ()->audioStatusChangedInternal(
						audioThread,
						active,
						formatInfo,
						errMsg ) );
  }


  @Override
  public void errorOccured( String text, Exception ex )
  {
    fireShowErrorMsg( text, ex );
  }


  @Override
  public void fileRead(
		String             format,
		int                begAddr,
		int                dataLen,
		int                startAddr,
		String             fileName,
		String             infoText,
		List<String>       fileExts,
		Map<String,byte[]> fileExt2Bytes,
		String             logText,
		boolean            errState )
  {
    EventQueue.invokeLater( ()->fileReadInternal(
					format,
					begAddr,
					dataLen,
					startAddr,
					fileName,
					infoText,
					fileExts,
					fileExt2Bytes,
					logText,
					errState ) );
  }


  @Override
  public void setVolumeLimits( int minLimit, int maxLimit )
  {
    this.volumeBar.setVolumeLimits( minLimit, maxLimit );
  }


  @Override
  public void updProgressPercent( int percent )
  {
    EventQueue.invokeLater( ()->updProgressPercentInternal( percent ) );
  }


  @Override
  public void updReadActivity( final String text )
  {
    EventQueue.invokeLater( ()->updReadActivityInternal( text ) );
  }


  @Override
  public void updVolume( int value )
  {
    this.volumeBar.updVolume( value );
  }


	/* --- Aktionen --- */

  private void doAbout()
  {
    JLabel label = new JLabel( Main.APPNAME );
    label.setFont( new Font( Font.SANS_SERIF, Font.BOLD, 18 ) );

    Object[] msg = new Object[] {
			label,
			"Version " + Main.VERSION + "\n\n"
				+ Main.COPYRIGHT
				+ "\n\n"
				+ "Lizenz:\n"
				+ "GNU General Public License (GNU-GPL)"
				+ " Version 3\n\n"
				+ "Die Nutzung dieser Software erfolgt"
				+ " auf eigenes Risiko.\n"
				+ "Jegliche Gew\u00E4hrleistung und Haftung"
				+ " ist ausgeschlossen!" };

    String title = "\u00DCber...";

    Image  iconImage = UIUtil.getLargestIconImage();
    if( iconImage != null ) {
      JOptionPane.showMessageDialog(
		this,
		msg,
		title,
		JOptionPane.INFORMATION_MESSAGE,
		new ImageIcon( iconImage ) );
    } else {
      JOptionPane.showMessageDialog(
		this,
		msg,
		title,
		JOptionPane.INFORMATION_MESSAGE );
    }
  }


  private void doAnalysisShow()
  {
    if( this.analysisFrm == null ) {
      this.analysisFrm = new AnalysisFrm();
    }
    UIUtil.toFront( this.analysisFrm );
    fireShowAnalysisData();
  }


  private void doAudioClose()
  {
    AudioThread audioThread = this.audioThread;
    if( audioThread != null ) {
      audioThread.fireStop();
    }
    setAudioStatus( false );
  }


  private void doAudioFileConcat()
  {
    if( this.audioFileConcatFrm == null ) {
      this.audioFileConcatFrm = new AudioFileConcatFrm();
    }
    UIUtil.toFront( this.audioFileConcatFrm );
  }


  private void doAudioOpenFile()
  {
    RecognitionSettings recSettings = checkAndGetRecognitionSettings();
    if( (recSettings != null) && (this.audioThread == null) ) {
      File file = UIUtil.showOpenFileDlg(
			this,
			"Audiodatei \u00F6ffnen",
			this.recentAudioDir,
			UIUtil.getAudioFileFilter() );
      if( file != null ) {
	if( openAudioFile( recSettings, file, -1 ) ) {
	  this.recentAudioDir = file.getParentFile();
	}
      }
    }
  }


  private void doAudioOpenLine()
  {
    RecognitionSettings recSettings = checkAndGetRecognitionSettings();
    if( (recSettings != null) && (this.audioThread == null) ) {

      // Dialog fuer Geraete- und Kanalauswahl
      JPanel             panel = new JPanel( new GridBagLayout() );
      GridBagConstraints gbc   = new GridBagConstraints(
					0, 0,
					GridBagConstraints.REMAINDER, 1,
					1.0, 1.0,
					GridBagConstraints.CENTER,
					GridBagConstraints.BOTH,
					new Insets( 5, 5, 5, 5 ),
					0, 0 );

      List<Mixer.Info> mixerInfos = AudioUtil.getTargetDataLineMixerInfo();
      JList<String> list = createMixerSelectionList(
		mixerInfos,
		UIUtil.getAudioPlayMixerName() );
      panel.add( new JScrollPane( list ), gbc );

      ChannelSelectFld channelFld = new ChannelSelectFld(
		"Sofern der Audiokanal in Stereo"
			+ " ge\u00F6ffnet werden kann:\n"
			+ "Von welchem Kanal soll gelesen werden?",
		this.recentChannel );
      gbc.fill    = GridBagConstraints.NONE;
      gbc.weightx = 0.0;
      gbc.weightx = 1.0;
      gbc.gridy++;
      panel.add( channelFld, gbc );

      if( JOptionPane.showConfirmDialog(
		  this,
		  panel,
		  "Auswahl Audioger\u00E4t",
		  JOptionPane.OK_CANCEL_OPTION,
		  JOptionPane.PLAIN_MESSAGE ) == JOptionPane.OK_OPTION )
      {
	int selectedIdx = list.getSelectedIndex();
	if( selectedIdx < 0 ) {
	  selectedIdx = 0;
	}
	Mixer.Info mixerInfo = null;
	if( (selectedIdx > 0) && (selectedIdx <= mixerInfos.size()) ) {
	  mixerInfo = mixerInfos.get( selectedIdx - 1 );
	}
	openAudioLine(
		recSettings,
		mixerInfo,
		channelFld.getSelectedChannel() );
      }
    }
  }


  private void doAudioReopen()
  {
    RecognitionSettings recSettings = checkAndGetRecognitionSettings();
    if( (recSettings != null) && (this.audioThread == null) ) {
      Object audioSrc = this.recentAudioSrc;
      if( audioSrc != null ) {
	if( audioSrc instanceof File ) {
	  openAudioFile(
		recSettings,
		(File) audioSrc,
		this.recentChannel );
	} else if( audioSrc instanceof Mixer.Info ) {
	  openAudioLine(
		recSettings,
		(Mixer.Info) audioSrc,
		this.recentChannel );
	}
      } else {
	openAudioLine( recSettings, null, this.recentChannel );
      }
    }
  }


  private void doFileFindDoublets()
  {
    List<Integer> modelRows = new ArrayList<>();

    this.fileTable.clearSelection();
    int nRows = this.fileTableModel.getRowCount();
    if( nRows > 1 ) {
      for( int i = 0; i < (nRows - 1); i++ ) {
	FileData fileData1 = this.fileTableModel.getRow( i );
	if( fileData1 != null ) {
	  for( int k = i + 1; k < nRows; k++ ) {
	    FileData fileData2 = this.fileTableModel.getRow( k );
	    if( fileData2 != null ) {
	      if( fileData1.equalsActualFileData( fileData2 ) ) {
		int row = this.fileTable.convertRowIndexToView( k );
		if( row >= 0 ) {
		  this.fileTable.addRowSelectionInterval( row, row );
		  modelRows.add( k );
		}
	      }
	    }
	  }
	}
      }
    }
    int nFound = modelRows.size();
    if( nFound > 0 ) {
      String msg = "Es wurde eine Dublette gefunden und markiert.\n"
			+ "M\u00F6chten Sie die Dublette jetzt entfernen?";
      if( nFound != 1 ) {
	msg = String.format(
		"Es wurden %d Dubletten gefunden und markiert.\n"
			+ "M\u00F6chten Sie die Dubletten jetzt entfernen?",
		nFound );
      }
      if( JOptionPane.showConfirmDialog(
		this,
		msg,
		UIUtil.TITLE_CONFIRMATIOn,
		JOptionPane.YES_NO_OPTION,
		JOptionPane.QUESTION_MESSAGE )
				== JOptionPane.YES_OPTION )
      {
	removeModelRows( modelRows );
      }
    } else {
      JOptionPane.showMessageDialog(
		this,
		"Keine Dubletten gefunden",
		"Dublettensuche",
		JOptionPane.INFORMATION_MESSAGE );
    }
  }


  private void doFileRemove()
  {
    int[] rows = this.fileTable.getSelectedRows();
    if( rows != null ) {
      if( rows.length > 0 ) {
	boolean       status    = true;
	List<Integer> modelRows = new ArrayList<>();
	for( int rowIdx : rows ) {
	  int modelIdx = this.fileTable.convertRowIndexToModel( rowIdx );
	  if( modelIdx >= 0 ) {
	    modelRows.add( modelIdx );
	    FileData fileData = this.fileTableModel.getRow( modelIdx );
	    if( fileData != null ) {
	      if( !fileData.isSaved() ) {
		status = false;
	      }
	    }
	  }
	}
	if( !status ) {
	  if( JOptionPane.showConfirmDialog(
			this,
			"Es wurden noch nicht alle markierten Dateien"
				+ " gespeichert.\n"
				+ "M\u00F6chten Sie trotzdem diese"
				+ " Dateien entfernen?",
			UIUtil.TITLE_CONFIRMATIOn,
			JOptionPane.YES_NO_OPTION,
			JOptionPane.WARNING_MESSAGE )
				== JOptionPane.YES_OPTION )
	  {
	    status = true;
	  }
	}
	if( status ) {
	  removeModelRows( modelRows );
	}
      }
    }
  }


  private void doFileRemoveAll()
  {
    if( !this.fileTableModel.isEmpty() ) {
      boolean status = checkAllFilesSaved();
      if( !status ) {
	if( JOptionPane.showConfirmDialog(
			this,
			"Es wurden noch nicht alle Dateien"
				+ " gespeichert.\n"
				+ "M\u00F6chten Sie trotzdem alle"
				+ " Dateien entfernen?",
			UIUtil.TITLE_CONFIRMATIOn,
			JOptionPane.YES_NO_OPTION,
			JOptionPane.WARNING_MESSAGE )
				== JOptionPane.YES_OPTION )
	{
	  status = true;
	}
      }
      if( status ) {
	this.fileTableModel.clear();
	updFileActionsEnabled();
      }
    }
  }


  private void doFileRemoveErr()
  {
    int nRows = this.fileTableModel.getRowCount();
    if( nRows > 0 ) {
      List<Integer> modelRows = new ArrayList<>();
      for( int i = 0; i < nRows; i++ ) {
	FileData fileData = this.fileTableModel.getRow( i );
	if( fileData != null ) {
	  if( fileData.hasError() ) {
	    modelRows.add( i );
	  }
	}
      }
      nRows = modelRows.size();
      if( nRows > 0 ) {
	if( JOptionPane.showConfirmDialog(
			this,
			"M\u00F6chten Sie die fehlerhaft eingelesenen"
				+ " Dateien entfernen?",
			UIUtil.TITLE_CONFIRMATIOn,
			JOptionPane.YES_NO_OPTION,
			JOptionPane.QUESTION_MESSAGE )
				== JOptionPane.YES_OPTION )

	removeModelRows( modelRows );
      } else {
	JOptionPane.showMessageDialog(
		this,
		"Es sind keine fehlerhaften Dateien enthalten.",
		"Hinweis",
		JOptionPane.INFORMATION_MESSAGE );
      }
    }
  }


  private void doFileSaveAs()
  {
    int[] rows = this.fileTable.getSelectedRows();
    if( rows != null ) {
      boolean cancelled = false;
      for( int row : rows ) {
	if( cancelled ) {
	  cancelled = false;
	  if( JOptionPane.showConfirmDialog(
			this,
			"Es sind noch weitere Dateien zum Speichern"
				+ " markiert.\n"
				+ "M\u00F6chten Sie diese noch speichern?",
			UIUtil.TITLE_CONFIRMATIOn,
			JOptionPane.YES_NO_OPTION,
			JOptionPane.QUESTION_MESSAGE )
					!= JOptionPane.YES_NO_OPTION )
	  {
	    break;
	  }
	}
	int modelRow = this.fileTable.convertRowIndexToModel( row );
	if( modelRow >= 0 ) {
	  FileData fileData = this.fileTableModel.getRow( modelRow );
	  if( fileData != null ) {

	    /*
	     * Dateifilter erstellen,
	     * dabei den bevorzugten an die erste Stelle setzen
	     */
	    String                 firstExt    = null;
	    Map<String,FileFilter> ext2Filter  = new HashMap<>();
	    List<FileFilter>       fileFilters = new ArrayList<>();
	    for( String fileExt : fileData.getFileExtensions() ) {
	      if( firstExt == null ) {
		firstExt = fileExt;
	      }
	      String     lowerExt = fileExt.toLowerCase();
	      FileFilter ff       = new FileNameExtensionFilter(
						String.format(
							"%s-Dateien (*.%s)",
							fileExt,
							lowerExt ),
						lowerExt );
	      fileFilters.add( ff );
	      ext2Filter.put( fileExt, ff );
	    }
	    String prefExt = null;
	    String format  = fileData.getFormat();
	    if( format != null ) {
	      prefExt = this.recentFormat2FileExt.get( format );
	    }
	    if( prefExt == null ) {
	      prefExt = firstExt;
	    }
	    if( prefExt != null ) {
	      FileFilter ff = ext2Filter.get( prefExt );
	      if( ff != null ) {
		fileFilters.remove( ff );
		fileFilters.add( 0, ff );
	      }
	    }

	    /*
	     * vorausgewaehlten Dateinamen erzeugen,
	     * dabei beim BIN-Format Adressen an den Dateinamen anhaengen
	     */
	    String fileName = fileData.getFileName();
	    if( (fileName != null) && (prefExt != null) ) {
	      if( !fileName.isEmpty() ) {
		StringBuilder buf = new StringBuilder();
		buf.append( fileName );
		if( prefExt.equals( "BIN" ) ) {
		  int begAddr = fileData.getBegAddr();
		  if( (begAddr > 0) && (begAddr < 0xFFFF) ) {
		    buf.append( String.format( "_%04X", begAddr ) );
		    int dataLen = fileData.getDataLength();
		    if( (dataLen > 0) && ((begAddr + dataLen) < 0x10000) ) {
		      buf.append( String.format(
					"_%04X",
					begAddr + dataLen - 1 ) );
		      int startAddr = fileData.getStartAddr();
		      if( (startAddr != 0) && (startAddr != 0xFFFF) ) {
			buf.append( String.format( "_%04X", startAddr ) );
		      }
		    }
		  }
		}
		fileName = buf.toString();
	      }
	    }

	    // Dateiauswahldialog
	    File file = UIUtil.showSaveFileDlg(
					this,
					"Eingelesene Datei speichern",
					this.recentSaveDir,
					fileName,
					null,
					fileFilters );
	    if( file != null ) {
	      try {
		fileData.save( this, file );
		this.recentSaveDir = file.getParentFile();
		this.fileTableModel.fireTableRowsUpdated(
							modelRow,
							modelRow );

		// Dateiendung als bevorzugte merken
		String s = file.getName();
		if( (s != null) && (format != null) ) {
		  int pos = s.lastIndexOf( '.' );
		  if( (pos + 1) > s.length() ) {
		    this.recentFormat2FileExt.put(
				format,
				s.substring( pos + 1 ).toUpperCase() );
		  }
		}
	      }
	      catch( IOException ex ) {
		showErrorMsg( null, ex );
	      }
	    } else {
	      cancelled = true;
	    }
	  }
	}
      }
    }
  }


  private void doFileShowContent()
  {
    int[] rows = this.fileTable.getSelectedRows();
    if( rows.length > 0 ) {
      List<FileData> files = new ArrayList<>( rows.length );
      for( int viewRow : rows ) {
	if( viewRow >= 0 ) {
	  int modelIdx = this.fileTable.convertRowIndexToModel( viewRow );
	  if( modelIdx >= 0 ) {
	    FileData fileData = this.fileTableModel.getRow( modelIdx );
	    if( fileData != null ) {
	      files.add( fileData );
	    }
	  }
	}
      }
      if( !files.isEmpty() ) {
	if( this.hexAsciiDiffFrm == null ) {
	  this.hexAsciiDiffFrm = new HexAsciiDiffFrm();
	}
	this.hexAsciiDiffFrm.setFiles( files );
	UIUtil.toFront( this.hexAsciiDiffFrm );
      }
    }
  }


  private void doFileShowLog()
  {
    FileData fileData = getFileDataByViewRowIdx(
					this.fileTable.getSelectedRow() );
    if( fileData != null ) {
      JTextArea textArea = new JTextArea( fileData.getLogText(), 12, 50 );
      textArea.setEditable( false );
      JOptionPane.showMessageDialog(
			this,
			new JScrollPane( textArea ),
			"Protokoll",
			JOptionPane.PLAIN_MESSAGE );
    }
  }


  private void doHelp( String page )
  {
    if( this.helpFrm == null ) {
      this.helpFrm = new HelpFrm();
    }
    this.helpFrm.setPage( page );
    UIUtil.toFront( this.helpFrm );
  }


  private void doQuit()
  {
    boolean status = checkAllFilesSaved();
    if( !status ) {
      if( JOptionPane.showConfirmDialog(
		this,
		"Es wurden noch nicht alle Dateien gespeichert.\n"
			+ "M\u00F6chten Sie trotzdem das Programm beenden?",
			UIUtil.TITLE_CONFIRMATIOn,
			JOptionPane.YES_NO_OPTION,
			JOptionPane.WARNING_MESSAGE )
					== JOptionPane.YES_OPTION )
      {
	status = true;
      }
    }
    if( status ) {
      AudioThread audioThread = this.audioThread;
      if( audioThread != null ) {
	audioThread.fireStop();
      }
      if( this.analysisFrm != null ) {
	this.analysisFrm.doClose();
      }
      if( this.helpFrm != null ) {
	this.helpFrm.doClose();
      }
      if( this.hexAsciiDiffFrm != null ) {
	this.hexAsciiDiffFrm.doClose();
      }
      this.btnAudioOpenFile.removeActionListener( this );
      this.btnAudioReopen.removeActionListener( this );
      this.btnAudioClose.removeActionListener( this );
      this.mnuAbout.removeActionListener( this );
      this.mnuAnalysisShow.removeActionListener( this );
      this.mnuAudioFileConcat.removeActionListener( this );
      this.mnuAudioOpenLine.removeActionListener( this );
      this.mnuAudioOpenFile.removeActionListener( this );
      this.mnuAudioClose.removeActionListener( this );
      this.mnuFileFindDoublets.removeActionListener( this );
      this.mnuFileRemove.removeActionListener( this );
      this.mnuFileRemoveAll.removeActionListener( this );
      this.mnuFileSaveAs.removeActionListener( this );
      this.mnuFileShowContent.removeActionListener( this );
      this.mnuFileShowLog.removeActionListener( this );
      this.mnuHelpHome.removeActionListener( this );
      this.mnuLicense.removeActionListener( this );
      this.mnuQuit.removeActionListener( this );
      this.mnuSettingsAudio.removeActionListener( this );
      if( this.mnuSettingsDelete != null ) {
	this.mnuSettingsDelete.removeActionListener( this );
      }
      if( this.mnuSettingsSave != null ) {
	this.mnuSettingsSave.removeActionListener( this );
      }
      this.popupFileRemove.removeActionListener( this );
      this.popupFileRemoveAll.removeActionListener( this );
      this.popupFileSaveAs.removeActionListener( this );
      this.popupFileShowContent.removeActionListener( this );
      this.popupFileShowLog.removeActionListener( this );
      this.cbMonitor.removeActionListener( this );
      this.sliderMonitorVolume.removeChangeListener( this );
      this.btnLoadParams.removeActionListener( this );
      this.fileTable.removeMouseListener( this );
      this.fileScrollPane.removeMouseListener( this );
      if( this.fileSelectionModel != null ) {
	this.fileSelectionModel.removeListSelectionListener( this );
      }
      this.dropTarget1.setActive( false );
      this.dropTarget2.setActive( false );
      this.dropTarget3.setActive( false );
      setVisible( false );
      dispose();
      if( audioThread != null ) {
	try {
	  audioThread.interrupt();
	  audioThread.join( 500 );
	}
	catch( InterruptedException ex ) {}
      }
      System.exit( 0 );
    }
  }


  private void doSettingsAudio()
  {
    JList<String> list = createMixerSelectionList(
				AudioUtil.getSourceDataLineMixerInfo(),
				UIUtil.getAudioPlayMixerName() );

    if( JOptionPane.showConfirmDialog(
		this,
		new JScrollPane( list ),
		"Audiowiedergabeger\u00E4t festlegen",
		JOptionPane.OK_CANCEL_OPTION,
		JOptionPane.PLAIN_MESSAGE ) == JOptionPane.OK_OPTION )
    {
      String mixerName = null;
      if( list.getSelectedIndex() > 0 ) {
	mixerName = list.getSelectedValue();
      }
      UIUtil.setAudioPlayMixerName( mixerName );
    }
  }


  private void doSettingsDelete()
  {
    boolean deleted      = false;
    File    settingsFile = Main.getSettingsFile();
    if( settingsFile != null ) {
      if( JOptionPane.showConfirmDialog(
		this,
		"Gespeicherte Einstellungen l\u00F6schen?",
		"Best\u00E4tigung",
		JOptionPane.YES_NO_OPTION ) == JOptionPane.YES_OPTION )
      {
	if( settingsFile.delete() ) {
	  deleted = true;
	}
      }
    } else {
      deleted = true;
    }
    if( this.mnuSettingsDelete != null ) {
      this.mnuSettingsDelete.setEnabled( !deleted );
    }
  }


  private void doSettingsSave()
  {
    File settingsFile = Main.getSettingsFile();
    if( settingsFile != null ) {
      Properties props = Main.getProperties();
      props.setProperty( PROP_VERSION, Main.VERSION );
      props.setProperty(
		PROP_ANALYSIS_ENABLED,
		String.valueOf( this.mnuAnalysisRec.isSelected() ) );
      props.setProperty(
		PROP_MONITOR_ENABLED,
		String.valueOf( this.cbMonitor.isSelected() ) );

      String value = "";
      Object fmt   = this.comboRecFmt.getSelectedItem();
      if( fmt != null ) {
	if( fmt instanceof RecognitionSettings.Format ) {
	  value = ((RecognitionSettings.Format) fmt).name();
	}
      }
      props.setProperty( PROP_FORMAT, value );
      UIUtil.getTableLayout( this.fileTable, props, PROP_PREFIX );
      UIUtil.getWindowBounds( this, props, PROP_PREFIX );
      if( this.analysisFrm != null ) {
	this.analysisFrm.getSettings( props );
      }
      if( this.audioFileConcatFrm != null ) {
	this.audioFileConcatFrm.getSettings( props );
      }
      AudioPlayFrm.getWindowSettings( props );
      if( this.helpFrm != null ) {
	this.helpFrm.getSettings( props );
      }
      if( this.hexAsciiDiffFrm != null ) {
	this.hexAsciiDiffFrm.getSettings( props );
      }
      boolean saved = false;
      try( OutputStream out = new BufferedOutputStream(
				new FileOutputStream( settingsFile ) ) )
      {
	props.storeToXML( out, Main.APPNAME + " Settings" );
	saved = true;
      }
      catch( Exception ex ) {
	saved = false;
	showErrorMsg( null, ex );
      }
      if( saved ) {
	if( this.mnuSettingsDelete != null ) {
	  this.mnuSettingsDelete.setEnabled( true );
	}
	if( !this.suppressSettingsSavedMsg ) {
	  JCheckBox cb = new JCheckBox(
				"Diese Meldung nicht mehr anzeigen." );
	  Object[] msg = new Object[] {
		"Einstellungen gespeichert in:\n" + settingsFile.getPath(),
		cb };
	  JOptionPane.showMessageDialog(
		this,
		msg,
		"Information",
		JOptionPane.INFORMATION_MESSAGE );
	  this.suppressSettingsSavedMsg = cb.isSelected();
	}
      }
    }
  }


	/* --- private Methoden --- */

  private void audioStatusChangedInternal(
				AudioThread audioThread,
				boolean     active,
				String      formatInfo,
				String      errMsg )
  {
    if( audioThread == this.audioThread ) {
      boolean stateRecent = false;
      if( active ) {
	this.fldAudioInfo.setText( formatInfo != null ? formatInfo : "" );
      }
      setAudioStatus( active );
      if( errMsg != null ) {
	showErrorMsg( errMsg );
      }
    }
  }


  private boolean checkAllFilesSaved()
  {
    boolean status = true;
    int     n      = this.fileTableModel.getRowCount();
    if( n > 0 ) {
      status = true;
      for( int i = 0; i < n; i++ ) {
	FileData fileData = this.fileTableModel.getRow( i );
	if( fileData != null ) {
	  if( !fileData.isSaved() ) {
	    status = false;
	    break;
	  }
	}
      }
    }
    return status;
  }


  private RecognitionSettings checkAndGetRecognitionSettings()
  {
    RecognitionSettings        settings = null;
    RecognitionSettings.Format fmt      = getSelectedRecFormat();
    if( fmt == null ) {
      showErrorMsg( "Bitte Aufnahmeformat ausw\u00E4hlen!" );
    }
    if( fmt != null ) {
      Float steepFlankStrobe = null;
      if( LoadParamDlg.getSteepFlanksProperty() ) {
	steepFlankStrobe = Float.valueOf(
				LoadParamDlg.getFlankStrokeProperty() );
      }
      settings = new RecognitionSettings(
			fmt,
			LoadParamDlg.getMinLevelProperty(),
			LoadParamDlg.getToleranceProperty(),
			steepFlankStrobe );
    }
    return settings;
  }


  private JList<String> createMixerSelectionList(
				List<Mixer.Info> mixerInfos,
				String           recentMixerName )
  {
    int      idx         = 0;
    int      idxToSelect = 0;
    String[] items       = new String[ mixerInfos.size() + 1 ];
    items[ idx++ ]       = "Standard";
    for( Mixer.Info mInfo : mixerInfos ) {
      String s = mInfo.getName();
      if( s != null ) {
	items[ idx ] = s;
	if( recentMixerName != null ) {
	  if( recentMixerName.equals( s ) ) {
	    idxToSelect = idx;
	  }
	}
      }
      idx++;
    }
    JList<String> list = new JList<>( items );
    list.setSelectionMode( ListSelectionModel.SINGLE_SELECTION );
    list.setSelectedIndex( idxToSelect );
    idx++;
    list.setVisibleRowCount( idx > 10 ? 10 : idx );
    return list;
  }


  private void fileReadInternal(
		String             format,
		int                begAddr,
		int                dataLen,
		int                startAddr,
		String             fileName,
		String             infoText,
		List<String>       fileExts,
		Map<String,byte[]> fileExt2Bytes,
		String             logText,
		boolean            errState )
  {
    this.fileTableModel.addRow(
			new FileData(
				format,
				begAddr,
				dataLen,
				startAddr,
				fileName,
				infoText,
				fileExts,
				fileExt2Bytes,
				logText,
				errState ) );
    updFileActionsEnabled();
  }


  private void fireShowAnalysisData()
  {
    EventQueue.invokeLater( ()->showAnalysisData() );
  }


  private void fireShowErrorMsg( final String text, final Exception ex )
  {
    EventQueue.invokeLater( ()->showErrorMsg( text, ex ) );
  }


  private FileData getFileDataByViewRowIdx( int viewIdx )
  {
    FileData fileData = null;
    if( viewIdx >= 0 ) {
      int modelIdx = this.fileTable.convertRowIndexToModel( viewIdx );
      if( modelIdx >= 0 ) {
	fileData = this.fileTableModel.getRow( modelIdx );
      }
    }
    return fileData;
  }


  private float getNormalizedMonitorVolume()
  {
    return UIUtil.getNormalizedValue( this.sliderMonitorVolume );
  }


  private RecognitionSettings.Format getSelectedRecFormat()
  {
    RecognitionSettings.Format fmt  = null;
    Object                     item = this.comboRecFmt.getSelectedItem();
    if( item != null ) {
      if( item instanceof RecognitionSettings.Format ) {
	fmt = (RecognitionSettings.Format) item;
      }
    }
    return fmt;
  }


  private boolean openAudioFile(
			RecognitionSettings recSettings,
			File                file,
			int                 requestedChannel )
  {
    boolean rv = false;
    if( this.audioThread == null ) {
      try {
	int channel  = 0;
	int channels = AudioSystem.getAudioFileFormat( file )
						.getFormat()
						.getChannels();
	if( (channels > 1)
	    && ((requestedChannel < 0) || (requestedChannel >= channels)) )
	{
	  ChannelSelectFld channelFld = new ChannelSelectFld(
		"Die Datei enth\u00E4lt mehrere Audiokan\u00E4le.\n"
			+ "Von welchem Kanal soll gelesen werden?",
		this.recentChannel );
	  if( JOptionPane.showConfirmDialog(
		  this,
		  channelFld,
		  "Auswahl Audiokanal",
		  JOptionPane.OK_CANCEL_OPTION,
		  JOptionPane.PLAIN_MESSAGE ) == JOptionPane.OK_OPTION )
	  {
	    channel = channelFld.getSelectedChannel();
	  } else {
	    channel = -1;
	  }
	}
	if( channel >= 0 ) {
	  setAudioStatus( true );
	  this.fldAudioSrc.setText( file.getPath() );
	  this.recentAudioSrc = file;
	  this.recentChannel  = channel;
	  this.audioThread    = new AudioFileThread(
		file,
		channel,
		recSettings,
		this.cbMonitor.isSelected(),
		UIUtil.getAudioPlayMixerName(),
		getNormalizedMonitorVolume(),
		this.mnuAnalysisRec.isSelected(),
		this );
	  this.audioThread.start();
	  rv = true;
	}
      }
      catch( IOException ex1 ) {
	showErrorMsg( null, ex1 );
      }
      catch( UnsupportedAudioFileException ex2 ) {
	showErrorMsg( AudioFileThread.TEXT_UNSUPPORTED_FILE_FORMAT, null );
      }
    }
    return rv;
  }


  private void openAudioLine(
			RecognitionSettings recSettings,
			Mixer.Info          mixerInfo,
			int                 requestedChannel )
  {
    setAudioStatus( true );
    this.fldAudioSrc.setText( mixerInfo != null ?
					mixerInfo.getName()
					: "Standard-Aufnahmeger\u00E4t" );
    UIUtil.setAudioPlayMixerName( 
		mixerInfo != null ? mixerInfo.getName() : null );

    this.recentAudioSrc = mixerInfo;
    this.recentChannel  = requestedChannel;
    this.audioThread    = new AudioLineThread(
					mixerInfo,
					requestedChannel,
					recSettings,
					this.mnuAnalysisRec.isSelected(),
					this );
    this.audioThread.start();
  }


  private void popupTriggered( MouseEvent e )
  {
    if( e.getButton() == MouseEvent.BUTTON3 ) {
      Component c = e.getComponent();
      if( (c == this.fileTable) || (c == this.fileScrollPane) ) {
	e.consume();
	this.popupFile.show( c, e.getX(), e.getY() );
      }
    }
  }


  private void removeModelRows( List<Integer> rows )
  {
    Collections.sort( rows );
    for( int i = rows.size() - 1; i >= 0; --i ) {
      this.fileTableModel.removeRow( rows.get( i ).intValue() );
    }
    updFileActionsEnabled();
  }


  private void requestedMonitorStateChanged()
  {
    AudioThread audioThread = this.audioThread;
    if( audioThread != null ) {
      audioThread.setMonitorEnabled(
			this.cbMonitor.isSelected(),
			UIUtil.getAudioPlayMixerName(),
			getNormalizedMonitorVolume() );
    }
    updMonitorFieldsEnabled();
  }


  private void setAudioStatus( boolean status )
  {
    boolean stateReopen      = false;
    boolean updVolumeEnabled = false;
    if( status ) {
      AudioThread audioThread = this.audioThread;
      if( audioThread != null ) {
	updVolumeEnabled = audioThread.isUpdVolumeEnabled();
      }
    } else {
      this.fldAudioInfo.setText( "" );
      this.audioThread = null;
      String srcText = this.fldAudioSrc.getText();
      if( srcText != null ) {
	stateReopen = !srcText.isEmpty();
      }
    }
    this.fldReadActivity.setText( "" );
    this.fldAudioSrc.setEnabled( status );
    this.labelAudioInfo.setEnabled( status );
    this.fldAudioInfo.setEnabled( status );
    this.labelRecFmt.setEnabled( !status );
    this.comboRecFmt.setEnabled( !status );
    this.btnLoadParams.setEnabled( !status );
    this.btnAudioOpenFile.setEnabled( !status );
    this.btnAudioReopen.setEnabled( stateReopen );
    this.btnAudioClose.setEnabled( status );
    this.labelReadActivity.setEnabled( updVolumeEnabled );
    if( status ) {
      if( updVolumeEnabled ) {
	this.labelProgress.setText( "Pegel:" );
	this.labelProgress.setEnabled( true );
      }
    } else {
      this.labelProgress.setText( LABEL_PROGRES_DEFAULT );
      this.labelProgress.setEnabled( false );
    }
    this.volumeBar.setVolumeBarState( updVolumeEnabled );
    updMonitorFieldsEnabled();
    this.mnuAudioOpenLine.setEnabled( !status );
    this.mnuAudioOpenFile.setEnabled( !status );
    this.mnuAudioClose.setEnabled( status );
    this.mnuSettingsLoadParams.setEnabled( !status );
    this.mnuSettingsBC.setEnabled( !status );
  }


  private void showAnalysisData()
  {
    if( (this.analysisFrm != null)
	&& (this.recentAudioFmt != null)
	&& (this.recentAudioData != null)
	&& (this.recentPhaseData != null) )
    {
      File   recentAudioFile = null;
      Object recentAudioSrc  = this.recentAudioSrc;
      if( recentAudioSrc != null ) {
	if( recentAudioSrc instanceof File ) {
	  recentAudioFile = (File) recentAudioSrc;
	}
      }
      this.analysisFrm.setAnalysisData(
				recentAudioFile,
				this.recentAudioFmt,
				this.recentAudioData,
				this.recentPhaseData );
    }
  }


  private void showErrorMsg( String text )
  {
    UIUtil.showErrorMsg( this, text );
  }


  private void showErrorMsg( String text, Exception ex )
  {
    UIUtil.showErrorMsg( this, text, ex );
  }


  private void updProgressPercentInternal( int percent )
  {
    if( this.audioThread != null ) {
      if( !this.progressBar.isEnabled() ) {
	this.labelProgress.setText( "Fortschritt:" );
	this.labelProgress.setEnabled( true );
	this.progressBar.setEnabled( true );
	this.progressBar.setMinimum( 0 );
	this.progressBar.setMaximum( 100 );
      }
      this.progressBar.setValue( percent );
    }
  }


  private void updReadActivityInternal( String text )
  {
    this.fldReadActivity.setText( text );
  }


  private void updFileActionsEnabled()
  {
    int     n     = this.fileTableModel.getRowCount();
    boolean state = (n > 0);
    this.mnuFileFindDoublets.setEnabled( n > 1 );
    this.mnuFileRemoveErr.setEnabled( state );
    this.mnuFileRemoveAll.setEnabled( state );
    this.popupFileRemoveErr.setEnabled( state );
    this.popupFileRemoveAll.setEnabled( state );
  }


  private void updMonitorFieldsEnabled()
  {
    boolean     state       = true;
    AudioThread audioThread = this.audioThread;
    if( audioThread != null ) {
      state = audioThread.supportsMonitoring();
    }
    this.cbMonitor.setEnabled( state );
    updMonitorVolumeSliderEnabled();
  }


  private void updMonitorVolumeSliderEnabled()
  {
    boolean state = this.cbMonitor.isSelected();
    if( state ) {
      AudioThread audioThread = this.audioThread;
      if( audioThread != null ) {
	state = audioThread.supportsMonitoring();
      }
    }
    this.sliderMonitorVolume.setEnabled( state );
  }
}
