/*
 * (c) 2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Fenster zur zur Anzeige des Inhalts von einer oder mehrere Dateien
 * in HEX-ASCII-Darstellung inklusive Markierung der Unterschiede
 */

package jkcload.ui;

import java.awt.EventQueue;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Properties;
import javax.swing.BorderFactory;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JViewport;
import jkcload.Main;


public class HexAsciiDiffFrm extends JFrame implements ActionListener
{
  private static final String PROP_PREFIX = Main.PROP_PREFIX + "hexascii.";
  private static final String DEFAULT_TITLE
			= Main.APPNAME + ": Dateiinhalte und Unterschiede";

  private boolean           notified;
  private List<FileData>    files;
  private JMenuItem         mnuClose;
  private JComboBox<String> comboFileExt;
  private HexAsciiDiffFld   hexAsciiDiffFld;
  private JScrollPane       scrollPane;
  private JLabel            labelStatus;


  public HexAsciiDiffFrm()
  {
    this.notified = false;
    this.files    = null;
    setDefaultCloseOperation( JFrame.DISPOSE_ON_CLOSE );
    UIUtil.setIconImagesAt( this );
    setTitle( DEFAULT_TITLE );


    // Menu
    JMenu mnuFile = new JMenu( "Datei" );
    mnuFile.setMnemonic( KeyEvent.VK_D );

    this.mnuClose = new JMenuItem( UIUtil.ITEM_CLOSE );
    mnuFile.add( this.mnuClose );

    JMenuBar mnuBar = new JMenuBar();
    mnuBar.add( mnuFile );
    setJMenuBar( mnuBar );


    // Fensterinhalt
    setLayout( new BorderLayout() );

    JPanel header = new JPanel( new FlowLayout( FlowLayout.LEFT, 5, 5 ) );
    header.add( new JLabel( "Dateityp:" ) );

    this.comboFileExt = new JComboBox<>();
    header.add( this.comboFileExt );
    add( header, BorderLayout.NORTH );

    this.hexAsciiDiffFld = new HexAsciiDiffFld();
    this.scrollPane      = new JScrollPane( this.hexAsciiDiffFld );
    add( this.scrollPane, BorderLayout.CENTER );

    this.labelStatus = new JLabel( UIUtil.STATUS_READY );
    this.labelStatus.setBorder(
		BorderFactory.createEmptyBorder( 5, 5, 5, 5 ) );
    add( this.labelStatus, BorderLayout.SOUTH );


    // Sonstiges
    if( !UIUtil.restoreWindowBounds(
			this,
			Main.getProperties(),
			PROP_PREFIX ) )
    {
      setSize( 500, 300 );
      setLocationByPlatform( true );
    }
    EventQueue.invokeLater( ()->this.hexAsciiDiffFld.requestFocus() );
  }


  public void getSettings( Properties props )
  {
    UIUtil.getWindowBounds( this, props, PROP_PREFIX );
  }


  public void doClose()
  {
    setVisible( false );
    dispose();
  }


  public void setFiles( List<FileData> files )
  {
    this.files = files;
    this.comboFileExt.removeAllItems();

    List<String> fileExtensions = null;
    for( FileData fileData : files ) {
      Collection<String> fe = fileData.getFileExtensions();
      if( fe != null ) {
	if( fileExtensions != null ) {
	  fileExtensions.retainAll( fe );
	} else {
	  fileExtensions = new ArrayList<>();
	  fileExtensions.addAll( fe );
	}
      }
    }
    if( fileExtensions != null ) {
      for( String fileExt : fileExtensions ) {
	this.comboFileExt.addItem( fileExt );
      }
    }
    fileExtChanged();

    String title  = DEFAULT_TITLE;
    int    nFiles = 0;
    if( files != null ) {
      nFiles = files.size();
      if( nFiles == 1 ) {
	StringBuilder buf = new StringBuilder( 128 );
	buf.append( Main.APPNAME );
	buf.append( " Dateiinhalt" );
	String fileName = files.get( 0 ).getFileName();
	if( fileName != null ) {
	  fileName = fileName.trim();
	  if( !fileName.isEmpty() ) {
	    buf.append( ": " );
	    buf.append( fileName );
	  }
	}
	title = buf.toString();
      }
    }
    setTitle( title );

    if( (nFiles > 1) && (this.comboFileExt.getItemCount() == 0) ) {
      EventQueue.invokeLater(
		()->UIUtil.showErrorMsg(
			this,
			"Die ausgew\u00E4hlten Dateien haben keinen"
			+ " gemeinsamen Dateityp,\n"
			+ "der angezeigt und verglichen werden"
			+ " k\u00F6nnte." ) );
    }
  }


	/* --- ActionListener --- */

  @Override
  public void actionPerformed( ActionEvent e )
  {
    Object src = e.getSource();
    if( src == this.mnuClose ) {
      doClose();
    } else if( src == this.comboFileExt ) {
      fileExtChanged();
    }
  }


	/* --- ueberschriebene Methoden --- */

  @Override
  public void addNotify()
  {
    super.addNotify();
    if( !this.notified ) {
      this.notified = true;
      this.mnuClose.addActionListener( this );
      this.comboFileExt.addActionListener( this );
    }
  }


  @Override
  public void removeNotify()
  {
    if( this.notified ) {
      this.notified = false;
      this.mnuClose.removeActionListener( this );
      this.comboFileExt.removeActionListener( this );
    }
    super.removeNotify();
  }


	/* --- private Methoden --- */

  private void fileExtChanged()
  {
    String   statusText = UIUtil.STATUS_READY;
    int      fileLen    = 0;
    byte[][] fileBytes  = null;
    String   fileExt    = null;
    Object   item       = this.comboFileExt.getSelectedItem();
    if( (item != null) && (this.files != null) ) {
      fileExt    = item.toString();
      int nFiles = this.files.size();
      if( (fileExt != null) && (nFiles > 0) ) {
	fileBytes = new byte[ nFiles ][];
	for( int i = 0; i < nFiles; i++ ) {
	  byte[] a = this.files.get( i ).getFileBytes( fileExt );
	  if( a != null ) {
	    if( a.length > fileLen ) {
	      fileLen = a.length;
	    }
	  }
	  fileBytes[ i ] = a;
	}
      }
    }

    // Dateien vergleichen
    int nDiff = 0;
    if( fileBytes != null ) {
      if( fileBytes.length > 1 ) {
	for( int filePos = 0; filePos < fileLen; filePos++ ) {
	  boolean differs = false;
	  byte    b       = (byte) 0;
	  for( int i = 0; i < fileBytes.length; i++ ) {
	    if( filePos >= fileBytes[ i ].length ) {
	      differs = true;
	      break;
	    }
	    if( i == 0 ) {
	      b = fileBytes[ i ][ filePos ];
	    } else if( fileBytes[ i ][ filePos ] != b ) {
	      differs = true;
	      break;
	    }
	  }
	  if( differs ) {
	    nDiff++;
	  }
	}
	if( nDiff > 0 ) {
	  statusText = String.format(
				"%d von %d Bytes unterschiedlich",
				nDiff,
				fileLen );
	} else {
	  /*
	   * Wenn die Dateien identisch sind,
	   * dann nur eine Datei anzeigen.
	   */
	  StringBuilder buf = new StringBuilder( 64 );
	  buf.append( "Alle " );
	  buf.append( fileBytes.length );
	  buf.append( '\u0020' );
	  if( fileExt != null ) {
	    if( !fileExt.isEmpty() ) {
	      buf.append( fileExt );
	      buf.append( '-' );
	    }
	  }
	  buf.append( "Dateien sind identisch." );
	  statusText     = buf.toString();
	  byte[] m       = fileBytes[ 0 ];
	  fileBytes      = new byte[ 1 ][];
	  fileBytes[ 0 ] = m;
	}
      }
    }

    // Anzeige aktualisieren
    this.hexAsciiDiffFld.setFileBytes( fileBytes );
    this.scrollPane.setColumnHeaderView(
			this.hexAsciiDiffFld.createColumnHeader() );
    this.scrollPane.setRowHeaderView(
			this.hexAsciiDiffFld.createRowHeader() );
    JViewport vp = this.scrollPane.getViewport();
    if( vp != null ) {
      vp.setViewPosition( new Point( 0, 0 ) );
    }
    this.labelStatus.setText( statusText );
  }
}
