/*
 * (c) 2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Komponente zur Anzeige des Inhalts von einer oder mehrere Dateien
 * in HEX-ASCII-Darstellung inklusive Markierung der Unterschiede
 */

package jkcload.ui;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.SystemColor;
import javax.swing.JComponent;
import javax.swing.Scrollable;
import javax.swing.SwingConstants;
import javax.swing.JTextArea;


public class HexAsciiDiffFld extends JComponent implements Scrollable
{
  private class ColumnHeader extends JComponent
  {
    private HexAsciiDiffFld hexAsciiDiffFld;

    private ColumnHeader( HexAsciiDiffFld hexAsciiDiffFld )
    {
      this.hexAsciiDiffFld = hexAsciiDiffFld;
    }

    @Override
    public Dimension getPreferredSize()
    {
      int rows = (this.hexAsciiDiffFld.getNumFiles() > 1 ? 2 : 1);
      return new Dimension(
			this.hexAsciiDiffFld.getPrefWidth(),
			(2 * HexAsciiDiffFld.MARGIN)
				+ (rows * this.hexAsciiDiffFld.hRow) );
    }

    @Override
    public void paintComponent( Graphics g )
    {
      g.setColor( SystemColor.controlText );

      int y      = this.hexAsciiDiffFld.yRow0;
      int x      = HexAsciiDiffFld.MARGIN;
      int nFiles = this.hexAsciiDiffFld.getNumFiles();
      if( nFiles > 1 ) {
	g.setFont(
		new Font(
			Font.SANS_SERIF,
			Font.BOLD,
			this.hexAsciiDiffFld.getFont().getSize() ) );
	for( int i = 1; i <= nFiles; i++ ) {
	  g.drawString( String.format( "Datei %d", i ), x, y );
	  x += this.hexAsciiDiffFld.wFileBlock;
	}
	y += this.hexAsciiDiffFld.hRow;
      }

      g.setFont( this.hexAsciiDiffFld.getFont() );
      x = HexAsciiDiffFld.MARGIN;;
      for( int i = 0; i < nFiles; i++ ) {
	for( int k = 0; k < 16; k++ ) {
	  g.drawString(
		String.format( "%02X", k ),
		x + (k * this.hexAsciiDiffFld.wHexCol),
		y );
	}
	for( int k = 0; k < 16; k++ ) {
	  g.drawString(
		String.format( "%1X", k ),
		x + this.hexAsciiDiffFld.wHexBlock
				+ (k * this.hexAsciiDiffFld.wAsciiChar),
		y );
	}
	x += this.hexAsciiDiffFld.wFileBlock;
      }
    }
  };


  private class RowHeader extends JComponent
  {
    private HexAsciiDiffFld hexAsciiDiffFld;

    private RowHeader( HexAsciiDiffFld hexAsciiDiffFld )
    {
      this.hexAsciiDiffFld = hexAsciiDiffFld;
    }

    @Override
    public Dimension getPreferredSize()
    {
      return new Dimension(
			(2 * HexAsciiDiffFld.MARGIN)
				+ (4 * this.hexAsciiDiffFld.wAsciiChar),
			this.hexAsciiDiffFld.getPrefHeight() );
    }

    @Override
    public void paintComponent( Graphics g )
    {
      g.setColor( SystemColor.controlText );
      g.setFont( this.hexAsciiDiffFld.getFont() );

      int len = this.hexAsciiDiffFld.fileLen;
      int y   = this.hexAsciiDiffFld.yRow0;
      for( int pos = 0; pos < len; pos += 16 ) {
	g.drawString(
		String.format( "%04X", pos ),
		HexAsciiDiffFld.MARGIN,
		y );
	y += this.hexAsciiDiffFld.hRow;
      }
    }
  };


  private static final int MARGIN      = 5;
  private static final int W_FILE_DIST = 20;

  private static final Color FG_EQUAL = Color.BLACK;
  private static final Color FG_DIFF  = Color.WHITE;
  private static final Color BG_DIFF  = Color.RED;

  private int      hFont;
  private int      hRow;
  private int      yRow0;
  private int      wAsciiChar;
  private int      wHexByte;
  private int      wHexBlock;
  private int      wHexCol;
  private int      wHexPreBG;
  private int      wFileBlock;
  private int      fileLen;
  private byte[][] fileBytes;
  private Font     font;


  public HexAsciiDiffFld()
  {
    this.wAsciiChar = 0;
    this.wHexByte   = 0;
    this.wHexBlock  = 0;
    this.wHexCol    = 0;
    this.wHexPreBG  = 0;
    this.wFileBlock = 0;
    this.fileLen    = 0;
    this.fileBytes  = null;

    this.hFont = 13;
    Font font  = new JTextArea().getFont();
    if( font != null ) {
      this.hFont = font.getSize();
    }
    this.font  = new Font( Font.MONOSPACED, Font.PLAIN, this.hFont );
    this.hRow  = this.hFont + 2;
    this.yRow0 = MARGIN + this.hRow;
    setPreferredSize( new Dimension( MARGIN, MARGIN ) );
  }


  public ColumnHeader createColumnHeader()
  {
    return new ColumnHeader( this );
  }


  public RowHeader createRowHeader()
  {
    return new RowHeader( this );
  }


  protected int getNumFiles()
  {
    return this.fileBytes != null ? this.fileBytes.length : 0;
  }


  protected int getPrefHeight()
  {
    int nRows = (this.fileLen + 15) / 16;
    return (2 * MARGIN) + (nRows * this.hRow);
  }


  protected int getPrefWidth()
  {
    int nFiles = getNumFiles();
    int wPref  = (2 * MARGIN) + (nFiles * this.wFileBlock);
    if( nFiles > 0 ) {
      wPref -= W_FILE_DIST;		// letzten Abstandshalter abziehen
    }
    return wPref;
  }


  public void setFileBytes( byte[][] fileBytes )
  {
    Component parent = getParent();
    if( parent != null ) {
      invalidate();
    }
    this.fileBytes = fileBytes;
    this.fileLen   = 0;
    if( this.fileBytes != null ) {
      for( byte[] a : this.fileBytes ) {
	if( a != null ) {
	  if( a.length > this.fileLen ) {
	    this.fileLen = a.length;
	  }
	}
      }
    }
    if( this.wAsciiChar > 0 ) {
      updPrefSize();
    }
    repaint();
  }


	/* --- Scrollable --- */

  @Override
  public Dimension getPreferredScrollableViewportSize()
  {
    return getPreferredSize();
  }


  @Override
  public int getScrollableBlockIncrement(
				Rectangle visibleRect,
				int       orientation,
				int       direction )
  {
    return orientation == SwingConstants.HORIZONTAL ?
						this.wFileBlock
						: visibleRect.height;
  }


  @Override
  public boolean getScrollableTracksViewportHeight()
  {
    return false;
  }


  @Override
  public boolean getScrollableTracksViewportWidth()
  {
    return false;
  }


  public int getScrollableUnitIncrement(
				Rectangle visibleRect,
				int       orientation,
				int       direction )
  {
    return orientation == SwingConstants.HORIZONTAL ?
						this.wAsciiChar
						: this.hRow;
  }


	/* --- ueberschriebene Methoden --- */

  @Override
  public Font getFont()
  {
    return this.font;
  }


  @Override
  public boolean isFocusable()
  {
    return true;
  }


  @Override
  public void paintComponent( Graphics g )
  {
    g.setFont( this.font );
    if( this.wAsciiChar <= 0 ) {
      FontMetrics fm = g.getFontMetrics();
      if( fm != null ) {
	this.wAsciiChar = fm.stringWidth( "W" );
	this.wHexByte   = fm.stringWidth( "FF" );
	this.wHexCol    = this.wHexByte + fm.stringWidth( "\u0020" );
	this.wHexPreBG  = (this.wHexCol - this.wHexByte) / 2;
	this.wHexBlock  = (this.wHexCol * 16) + this.wAsciiChar;
	this.wFileBlock = this.wHexBlock
				+ (this.wAsciiChar * 16)
				+ W_FILE_DIST;
	updPrefSize();
      }
    }

    Rectangle viewRect = UIUtil.getViewRect( this );
    if( (viewRect != null)
	&& (this.wAsciiChar > 0)
	&& (this.wHexByte > 0)
	&& (this.wHexCol > 0)
	&& (this.fileLen > 0)
	&& (this.fileBytes != null) )
    {
      // Hintergrund
      g.setColor( SystemColor.text );
      g.fillRect( viewRect.x, viewRect.y, viewRect.width, viewRect.height );

      // Dateiinhalte zeilenweise ausgeben
      int rowsToSkip = (viewRect.y - MARGIN) / this.hRow;
      if( rowsToSkip < 0 ) {
	rowsToSkip = 0;
      }
      int y       = this.yRow0 + (rowsToSkip * this.hRow);
      int filePos = rowsToSkip * 16;	// Position innerhalb der Datei
      while( filePos < this.fileLen ) {
	if( y > (viewRect.y + viewRect.height) ) {
	  break;
	}

	// Zeichnen einer Zeile
	for( int colIdx = 0; colIdx < 16; colIdx++ ) {
	  if( filePos >= this.fileLen ) {
	    break;
	  }

	  // pruefen, ob sich in den Dateien das Byte unterscheidet
	  boolean bDiffers = false;
	  byte    bFirst   = (byte) 0;
	  for( int fileIdx = 0;
	       fileIdx < this.fileBytes.length;
	       fileIdx++ )
	  {
	    if( filePos >= this.fileBytes[ fileIdx ].length ) {
	      bDiffers = true;
	      break;
	    }
	    if( fileIdx == 0 ) {
	      bFirst = this.fileBytes[ fileIdx ][ filePos ];
	    } else if( this.fileBytes[ fileIdx ][ filePos ] != bFirst ) {
	      bDiffers = true;
	      break;
	    }
	  }

	  // Byte fuer jede Datei ausgeben
	  int xHex   = colIdx * this.wHexCol;
	  int xAscii = this.wHexBlock + (colIdx * this.wAsciiChar);
	  for( int fileIdx = 0;
	       fileIdx < this.fileBytes.length;
	       fileIdx++ )
	  {
	    if( bDiffers ) {
	      // Hintergrund markieren
	      int yDiff = y - this.hFont + 2;
	      int hDiff = this.hFont + 2;
	      g.setColor( BG_DIFF );
	      g.fillRect(
			MARGIN + (fileIdx * this.wFileBlock) + xHex
					- this.wHexPreBG,
			yDiff,
			this.wHexCol,
			hDiff );
	      g.fillRect(
			MARGIN + (fileIdx * this.wFileBlock) + xAscii,
			yDiff,
			this.wAsciiChar,
			hDiff );
	    }
	    if( filePos < this.fileBytes[ fileIdx ].length ) {
	      int b = (int) this.fileBytes[ fileIdx ][ filePos ] & 0xFF;
	      g.setColor( bDiffers ? FG_DIFF : FG_EQUAL );
	      g.drawString(
			String.format( "%02X", b ),
			MARGIN + (fileIdx * this.wFileBlock) + xHex,
			y );
	      if( (b < 0x20) || (b > 0x7F) ) {
		b = '.';
	      }
	      g.drawString(
			Character.toString( (char) b ),
			MARGIN + (fileIdx * this.wFileBlock) + xAscii,
			y );
	    }
	  }
	  xHex += this.wHexCol;
	  xAscii += this.wAsciiChar;
	  filePos++;
	}
	y += this.hRow;
      }
    }
  }


	/* --- private Methoden --- */

  private void updPrefSize()
  {
    setPreferredSize(
		new Dimension(
			getPrefWidth(),
			getPrefHeight() ) );
  }
}
