/*
 * (c) 2023-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Fenster zur Anzeige der Hilfe
 */

package jkcload.ui;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.io.IOException;
import java.net.URL;
import java.util.EmptyStackException;
import java.util.Properties;
import java.util.Stack;
import javax.swing.JButton;
import javax.swing.JEditorPane;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JScrollPane;
import javax.swing.JToolBar;
import javax.swing.JViewport;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;
import javax.swing.event.MenuEvent;
import javax.swing.event.MenuListener;
import javax.swing.text.Document;
import jkcload.Main;


public class HelpFrm extends JFrame implements
					ActionListener,
					HyperlinkListener,
					MenuListener
{
  public static final String HOME_PAGE    = "/help/home.htm";
  public static final String LICENSE_PAGE = "/help/license.htm";


  private static class PageStackEntry
  {
    private URL    url;
    private Double viewPos;	// relative Angabe von 0.0 bis 1.0

    private PageStackEntry( URL url )
    {
      this.url     = url;
      this.viewPos = null;
    }
  };

  private static final String PROP_PREFIX = Main.PROP_PREFIX + "help.";

  private boolean               notified;
  private URL                   urlHome;
  private Stack<PageStackEntry> pageStack;
  private String                pageResource;
  private JButton               btnBack;
  private JButton               btnHome;
  private JMenu                 mnuEdit;
  private JMenuItem             mnuClose;
  private JMenuItem             mnuCopy;
  private JMenuItem             mnuSelectAll;
  private JEditorPane           editorPane;
  private JScrollPane           scrollPane;


  public HelpFrm()
  {
    this.notified  = false;
    this.pageStack = new Stack<>();
    this.urlHome   = getClass().getResource( HOME_PAGE );
    setDefaultCloseOperation( JFrame.DISPOSE_ON_CLOSE );
    setTitle( Main.APPNAME + " Hilfe" );


    // Fenster-Icons
    UIUtil.setIconImagesAt( this );


    // Menu
    JMenu mnuFile = new JMenu( "Datei" );
    mnuFile.setMnemonic( KeyEvent.VK_D );

    this.mnuClose = new JMenuItem( UIUtil.ITEM_CLOSE );
    mnuFile.add( this.mnuClose );

    this.mnuEdit = new JMenu( "Bearbeiten" );
    mnuEdit.setMnemonic( KeyEvent.VK_B );

    this.mnuCopy = new JMenuItem( "Kopieren" );
    this.mnuEdit.add( this.mnuCopy );
    this.mnuEdit.addSeparator();

    this.mnuSelectAll = new JMenuItem( "Alles ausw\u00E4hlen" );
    mnuEdit.add( this.mnuSelectAll );

    JMenuBar mnuBar = new JMenuBar();
    mnuBar.add( mnuFile );
    mnuBar.add( this.mnuEdit );
    setJMenuBar( mnuBar );


    // Fensterinhalt
    setLayout( new BorderLayout() );

    JToolBar toolBar = new JToolBar();
    toolBar.setFloatable( false );
    toolBar.setRollover( true );
    add( toolBar, BorderLayout.NORTH );

    this.btnBack = UIUtil.createImageButton(
					"/images/back.png",
					"Zur\u00FCck" );
    toolBar.add( this.btnBack );

    this.btnHome = UIUtil.createImageButton(
					"/images/home.png",
					"Startseite" );
    toolBar.add( this.btnHome );

    this.editorPane = new JEditorPane();
    this.editorPane.setEditable( false );
    this.editorPane.setPreferredSize( new Dimension( 500, 400 ) );

    this.scrollPane = new JScrollPane( editorPane );
    add( this.scrollPane, BorderLayout.CENTER );


    // Sonstiges
    if( !UIUtil.restoreWindowBounds(
			this,
			Main.getProperties(),
			PROP_PREFIX ) )
    {
      pack();
      setLocationByPlatform( true );
    }
  }


  public void doClose()
  {
    setVisible( false );
    dispose();
  }


  public void getSettings( Properties props )
  {
    UIUtil.getWindowBounds( this, props, PROP_PREFIX );
  }


  public void setPage( String page )
  {
    setPage(
	true,
	page != null ? getClass().getResource( page ) : null,
	null );
  }


	/* --- ActionListener --- */

  @Override
  public void actionPerformed( ActionEvent e )
  {
    Object src = e.getSource();
    if( src == this.btnBack ) {
      doBack();
    } else if( src == this.btnHome ) {
      doHome();
    } else if( src == this.mnuClose ) {
      doClose();
    } else if( src == this.mnuCopy ) {
      this.editorPane.copy();
      this.editorPane.requestFocus();
    } else if( src == this.mnuSelectAll ) {
      this.editorPane.selectAll();
      this.editorPane.requestFocus();
    }
  }


	/* --- HyperlinkListener --- */

  @Override
  public void hyperlinkUpdate( HyperlinkEvent e )
  {
    if( e.getEventType() == HyperlinkEvent.EventType.ACTIVATED )
      setPage( true, e.getURL(), null );
  }


	/* --- MenuListener --- */

  @Override
  public void menuCanceled( MenuEvent e )
  {
    // leer
  }


  @Override
  public void menuDeselected( MenuEvent e )
  {
    // leer
  }


  @Override
  public void menuSelected( MenuEvent e )
  {
    if( e.getSource() == this.mnuEdit ) {
      int begPos = this.editorPane.getSelectionStart();
      this.mnuCopy.setEnabled(
	(begPos >= 0) && (begPos < this.editorPane.getSelectionEnd()) );
    }
  }


	/* --- ueberschriebene Methoden --- */

  @Override
  public void addNotify()
  {
    super.addNotify();
    if( !this.notified ) {
      this.notified = true;
      this.btnBack.addActionListener( this );
      this.btnHome.addActionListener( this );
      this.mnuClose.addActionListener( this );
      this.mnuCopy.addActionListener( this );
      this.mnuSelectAll.addActionListener( this );
      this.mnuEdit.addMenuListener( this );
      this.editorPane.addHyperlinkListener( this );
    }
  }


  @Override
  public void removeNotify()
  {
    super.removeNotify();
    if( this.notified ) {
      this.notified = false;
      this.btnBack.removeActionListener( this );
      this.btnHome.removeActionListener( this );
      this.mnuClose.removeActionListener( this );
      this.mnuCopy.removeActionListener( this );
      this.mnuSelectAll.removeActionListener( this );
      this.mnuEdit.removeMenuListener( this );
      this.editorPane.removeHyperlinkListener( this );
    }
  }


	/* --- Aktionen --- */

  private void doBack()
  {
    if( this.pageStack.size() > 1 ) {
      try {
	this.pageStack.pop();	// aktuelle Seite vom Stack entfernen
	PageStackEntry entry = this.pageStack.pop();
	setPage( false, entry.url, entry.viewPos );
      }
      catch( EmptyStackException ex ) {}
    }
  }


  private void doHome()
  {
    this.pageStack.clear();
    setPage( HOME_PAGE );
  }


	/* --- private Methoden --- */

  private void setPage(
		boolean saveCurViewPos,
		URL     url,
		Double  viewPos )
  {
    if( url == null ) {
      this.pageStack.clear();
      url = this.urlHome;
    }
    if( url != null ) {

      /*
       * Seite nur anzeigen, wenn sie sich von der vorhergehenden
       * unterscheidet
       */
      boolean        alreadyVisible = false;
      PageStackEntry topEntry       = null;
      if( !this.pageStack.isEmpty() ) {
	topEntry = this.pageStack.peek();
	if( (url != null) && (topEntry.url != null) ) {
	  if( topEntry.url.equals( url ) ) {
	    alreadyVisible = true;
	  }
	}
      }
      if( !alreadyVisible ) {
	try {

	  // aktuelle Position ermitteln und merken
	  if( saveCurViewPos && (topEntry != null) ) {
	    topEntry.viewPos = null;
	    int h = this.editorPane.getHeight();
	    if( h > 0 ) {
	      JViewport vp = this.scrollPane.getViewport();
	      if( vp != null ) {
		Point pt = vp.getViewPosition();
		if( pt != null ) {
		  double d = (double) pt.y / (double) h;
		  if( (d > 0.0) && (d <= 1.0) ) {
		    topEntry.viewPos = d;
		  }
		}
	      }
	    }
	  }

	  // neue Seite anzeigen
	  if( url != null ) {
	    Document doc = this.editorPane.getDocument();
	    if( doc != null ) {
	      doc.putProperty( Document.StreamDescriptionProperty, null );
	    }
	    this.editorPane.setPage( url );
	  }

	  /*
	   * wenn Seite angezeigt werden konnte,
	   * dann neuen Stack-Eintrag erzeugen
	   */
	  this.pageStack.push( new PageStackEntry( url ) );

	  // Zurueck-Aktion aktualisieren
	  boolean stateBack = (this.pageStack.size() > 1);
	  this.btnBack.setEnabled( stateBack );
	}
	catch( Exception ex ) {
	  UIUtil.showErrorMsg(
		this,
		"Die Hilfeseite kann nicht angezeigt werden.",
		ex );
	}
      }
    }
  }
}
