/*
 * (c) 2023-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Daten einer Datei
 */

package jkcload.ui;

import java.awt.Component;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import javax.swing.JCheckBox;
import javax.swing.JOptionPane;


public class FileData
{
  private static boolean suppressLogInfo = false;

  private String             format;
  private int                begAddr;
  private int                dataLen;
  private int                startAddr;
  private String             begAddrText;
  private String             dataLenText;
  private String             startAddrText;
  private String             fileName;
  private List<String>       fileExts;
  private String             fileExtsText;
  private Map<String,byte[]> fileExt2Bytes;
  private String             infoText;
  private String             logText;
  private boolean            errState;
  private boolean            saved;


  public FileData(
		String             format,
		int                begAddr,
		int                dataLen,
		int                startAddr,
		String             fileName,
		String             infoText,
		List<String>       fileExts,
		Map<String,byte[]> fileExt2Bytes,
		String             logText,
		boolean            errState )
  {
    this.format        = format;
    this.begAddr       = begAddr;
    this.dataLen       = dataLen;
    this.startAddr     = startAddr;
    this.fileName      = trim( fileName );
    this.infoText      = trim( infoText );
    this.fileExts      = fileExts;
    if( fileExts.size() == 1 ) {
      this.fileExtsText = this.fileExts.get( 0 );
    } else {
      StringBuilder buf = new StringBuilder();
      for( String fileExt : fileExts ) {
	if( buf.length() > 0 ) {
	  buf.append( ", " );
	}
	buf.append( fileExt );
      }
      this.fileExtsText = buf.toString();
    }
    this.fileExt2Bytes = fileExt2Bytes;
    this.logText       = trim( logText );

    this.errState      = errState;
    this.saved         = false;

    // Anfangsadresse in textueller Form
    this.begAddrText = null;
    if( (begAddr > 0) && (begAddr < 0xFFFF) ) {
      this.begAddrText = String.format( "%04Xh", begAddr );
    }

    // Laenge in textueller Form
    this.dataLenText = String.format( "%04Xh", dataLen );

    // Startadresse in textueller Form
    this.startAddrText = null;
    if( (startAddr > 0) && (startAddr < 0xFFFF) ) {
      this.startAddrText = String.format( "%04Xh", startAddr );
    }
  }


  public boolean equalsActualFileData( FileData fileData )
  {
    return equals( this.format, fileData.format)
		&& (this.begAddr == fileData.begAddr)
		&& (this.dataLen == fileData.dataLen)
		&& (this.startAddr == fileData.startAddr)
		&& equals( this.fileName, fileData.fileName)
		&& equals( this.fileExtsText, fileData.fileExtsText)
		&& equals( this.fileExt2Bytes, fileData.fileExt2Bytes )
		&& equals( this.infoText, fileData.infoText)
		&& equals( this.logText, fileData.logText);
  }


  public int getBegAddr()
  {
    return this.begAddr;
  }


  public String getBegAddrText()
  {
    return this.begAddrText;
  }


  public int getDataLength()
  {
    return this.dataLen;
  }


  public String getDataLengthText()
  {
    return this.dataLenText;
  }


  public byte[] getFileBytes( String fileExt )
  {
    return this.fileExt2Bytes.get( fileExt.trim().toUpperCase() );
  }


  public List<String> getFileExtensions()
  {
    return this.fileExts;
  }


  public String getFileExtensionsText()
  {
    return this.fileExtsText;
  }


  public String getFileName()
  {
    return this.fileName;
  }


  public String getFormat()
  {
    return this.format;
  }


  public String getInfoText()
  {
    return this.infoText;
  }


  public String getLogText()
  {
    return this.logText;
  }


  public int getStartAddr()
  {
    return this.startAddr;
  }


  public String getStartAddrText()
  {
    return this.startAddrText;
  }


  public String getStatusText()
  {
    return this.errState ? "Fehler" : "OK";
  }


  public boolean hasError()
  {
    return this.errState;
  }


  public boolean hasLog()
  {
    return this.logText != null;
  }


  public boolean isSaved()
  {
    return this.saved;
  }


  public void save( Component owner, File file ) throws IOException
  {
    boolean saved = false;

    // zu speichernde Bytes anhand der Dateiendung ermitteln
    byte[] fileBytes    = null;
    String fileBaseName = null;
    String fileName     = file.getName();
    if( fileName != null ) {
      int pos = fileName.lastIndexOf( '.' );
      if( (pos + 1) < fileName.length() ) {
	fileBaseName = fileName.substring( 0, pos );
	fileBytes    = this.fileExt2Bytes.get(
		  	    fileName.substring( pos + 1 ).toUpperCase() );
      }
    }
    if( fileBytes == null ) {
      throw new IOException( "Die Datei kann in dem durch die Dateiendung"
			+ " angegebenen Format nicht gespeichert werden.\n"
			+ "Geben Sie bitte einen Dateinamen mit einer"
			+ " unterst\u00FCtzten Endung an!" );
    }

    // Datei speichern
    try( FileOutputStream out = new FileOutputStream( file ) ) {
      out.write( fileBytes );
    }
    saved = true;

    // ggf. Fehlerprotokoll speichern
    if( this.errState && (this.logText != null) && (fileBaseName != null) ) {
      if( !this.logText.isEmpty() ) {
	File dirFile       = file.getParentFile();
	String logFileName = fileBaseName += "-error.log";
	File logFile = null;
	if( dirFile != null ) {
	  logFile = new File( dirFile, logFileName );
	} else {
	  logFile = new File( logFileName );
	}
	try {
	  if( logFile.exists() ) {
	    throw new IOException( "Eine Datei mit dem gleichen Namen"
					+ " existiert bereits." );
	  }
	  try ( FileWriter writer = new FileWriter( logFile ) ) {
	    writer.write( "File:\n" );
	    writer.write( logFileName );
	    writer.write( "\n\n" );
	    writer.write( this.logText );
	    writer.close();
	  }
	  if( !suppressLogInfo ) {
	    JCheckBox cb = new JCheckBox(
				"Diesen Hinweis nicht mehr anzeigen" );
	    JOptionPane.showMessageDialog(
			owner,
			new Object[] {
				"Da die Datei nicht vollst\u00E4ndig bzw."
					+ " fehlerfrei eingelesen werden"
					+ " konnte,\n"
					+ "wurde eine Protokolldatei"
					+ " gespeichert unter:\n\n"
					+ logFile.getPath()
					+ "\n\n",
				cb },
			"Hinweis",
			JOptionPane.INFORMATION_MESSAGE );
	    suppressLogInfo = cb.isSelected();
	  }
	}
	catch( IOException ex ) {
	  saved = false;
	  StringBuilder buf = new StringBuilder( 256 );
	  buf.append( "Die zugeh\u00F6rige Fehlerprotokolldatei\n" );
	  buf.append( logFile.getPath() );
	  buf.append( "\nkonnte nicht gespeichert werden." );
	  String exMsg = ex.getMessage();
	  if( exMsg != null ) {
	    if( !exMsg.isEmpty() ) {
	      buf.append( "\n\n" );
	      buf.append( exMsg );
	    }
	  }
	  throw new IOException( buf.toString() );
	}
      }
    }
    this.saved = saved;
  }


	/* --- private Methoden --- */

  private static boolean equals( byte[] b1, byte[] b2 )
  {
    boolean rv = false;
    if( (b1 != null) && (b2 != null) ) {
      if( b1.length == b2.length ) {
	rv = true;
	for( int i = 0; i < b1.length; i++ ) {
	  if( b1[ i ] != b2[ i ] ) {
	    rv = false;
	    break;
	  }
	}
      }
    } else {
      int len1 = 0;
      int len2 = 0;
      if( b1 != null ) {
	len1 = b1.length;
      }
      if( b2 != null ) {
	len2 = b1.length;
      }
      if( (len1 == len2) && (len1 == 0) ) {
	rv = true;
      }
    }
    return rv;
  }


  private static boolean equals(
				Map<String,byte[]> m1,
				Map<String,byte[]> m2 )
  {
    boolean rv = false;
    if( (m1 != null) && (m2 != null) ) {
      if( m1.size() == m2.size() ) {
	rv = true;
	for( String k : m1.keySet() ) {
	  if( !equals( m1.get( k ), m2.get( k ) ) ) {
	    rv = false;
	    break;
	  }
	}
      }
    } else if( (m1 == null) && (m2 == null) ) {
      rv = true;
    }
    return rv;
  }


  private static boolean equals( String s1, String s2 )
  {
    if( s1 == null ) {
      s1 = "";
    }
    return s1.equals( s2 != null ? s2 : "" );
  }


  private static String trim( String text )
  {
    if( text != null ) {
      text = text.trim();
      if( text.isEmpty() ) {
	text = null;
      }
    }
    return text;
  }
}
