/*
 * (c) 2024-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Fenster zur Audiowiedergabe
 */

package jkcload.ui;

import java.awt.EventQueue;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.Properties;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.SourceDataLine;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JProgressBar;
import javax.swing.JSlider;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.Timer;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import jkcload.Main;
import jkcload.audio.AudioUtil;


public class AudioPlayFrm
			extends JFrame
			implements
				ActionListener,
				ChangeListener,
				Runnable
{
  private static final String PROP_PREFIX
				= Main.PROP_PREFIX + "audio.player.";

  private static final int PROGRESS_BAR_MAX       = 1000;
  private static final int VOLUME_SLIDER_MAX      = 1000;
  private static final int UPD_PLAYED_TIME_MILLIS = 200;
  private static final int UPD_POSITION_MILLIS    = 50;

  private static AudioPlayFrm instance           = null;
  private static Rectangle    recentWindowBounds = null;

  private AnalysisFrm       analysisFrm;
  private AudioFormat       audioFmt;
  private byte[]            audioData;
  private int               begFrameIdx;
  private int               curFrameIdx;
  private int               endFrameIdx;
  private int               frameLen;
  private int               frameRange;
  private int               frameRate;
  private int               progressValue;
  private volatile int      playedSeconds;
  private volatile boolean  playEnabled;
  private Thread            playThread;
  private Object            playLock;
  private Float             requestedVolume;
  private javax.swing.Timer updPlayedTimeTimer;
  private javax.swing.Timer updPositionTimer;
  private JButton           btnClose;
  private JProgressBar      progressBar;
  private JSlider           sliderVolume;
  private JTextField        fldPlayedTime;


  public static void close()
  {
    if( instance != null )
      instance.doClose();
  }


  public static void getWindowSettings( Properties props )
  {
    if( instance != null ) {
      UIUtil.getWindowBounds( instance, props, PROP_PREFIX );
    } else {
      UIUtil.getWindowBounds(
			recentWindowBounds,
			true,
			props,
			PROP_PREFIX );
    }
  }


  public static void open(
			AnalysisFrm analysisFrm,
			byte[]      audioData,
			int         frameIdx,
			int         frameLen,
			AudioFormat audioFmt )
  {
    if( instance != null ) {
      instance.doClose();
    }
    instance = new AudioPlayFrm(
				analysisFrm,
				audioData,
				frameIdx,
				frameLen,
				audioFmt );
    instance.setVisible( true );
  }


	/* --- ActionListener --- */

  @Override
  public void actionPerformed( ActionEvent e )
  {
    Object src = e.getSource();
    if( src == this.btnClose ) {
      doClose();
    } else if( src == this.updPlayedTimeTimer ) {
      updPlayedTimeFields();
    } else if( src == this.updPositionTimer ) {
      updPlayPosition();
    }
  }


	/* --- ChangeListener --- */

  @Override
  public void stateChanged( ChangeEvent e )
  {
    if( e.getSource() == this.sliderVolume ) {
      float volume = UIUtil.getNormalizedValue( this.sliderVolume );
      synchronized( this.playLock ) {
	this.requestedVolume = Float.valueOf( volume );
      }
      UIUtil.setAudioPlayVolume( volume );
    }
  }


	/* --- Runnable --- */

  @Override
  public void run()
  {
    this.playedSeconds = 0;

    int frameSize = this.audioFmt.getFrameSize();
    if( frameSize > 0 ) {
      Exception      audioEx = null;
      SourceDataLine line    = null;
      try {
	line = AudioUtil.openSourceDataLine(
				this.audioFmt,
				UIUtil.getAudioPlayMixerName() );

	int framePortionSize = this.frameRate * UPD_POSITION_MILLIS / 1000;
	int audioPortionSize = framePortionSize * frameSize;
	if( audioPortionSize > 0 ) {
	  while( this.playEnabled && (this.curFrameIdx < endFrameIdx) ) {
	    checkRequestedVolume( line );
	    int audioIdx = this.curFrameIdx * frameSize;
	    int audioLen = audioPortionSize;
	    if( (audioIdx + audioLen) > this.audioData.length ) {
	      audioLen = this.audioData.length - audioIdx;
	    }
	    if( audioLen > 0 ) {
	      line.write( this.audioData, audioIdx, audioLen );
	    }
	    this.curFrameIdx += framePortionSize;
	  }
	} else {
	  checkRequestedVolume( line );
	  int audioIdx = this.curFrameIdx * frameSize;
	  int audioLen = this.frameLen * frameSize;
	  if( (audioIdx + audioLen) > this.audioData.length ) {
	    audioLen = this.audioData.length - audioIdx;
	  }
	  if( audioLen > 0 ) {
	    line.write( this.audioData, audioIdx, audioLen );
	  }
	}
      }
      catch( Exception ex ) {
	audioEx = ex;
      }
      finally {
	if( line != null ) {
	  AudioUtil.closeSourceDataLine( line );
	}
	final Exception ex = audioEx;
	EventQueue.invokeLater( ()->playFinished( ex ) );
      }
    }
  }


	/* --- Konstruktor --- */

  private AudioPlayFrm(
		AnalysisFrm analysisFrm,
		byte[]      audioData,
		int         frameIdx,
		int         frameLen,
		AudioFormat audioFmt )
  {
    this.analysisFrm   = analysisFrm;
    this.audioData     = audioData;
    this.begFrameIdx   = frameIdx;
    this.curFrameIdx   = frameIdx;
    this.endFrameIdx   = this.begFrameIdx + frameLen;
    this.frameLen      = frameLen;
    this.frameRange    = this.endFrameIdx - frameIdx;
    this.frameRate     = Math.round( audioFmt.getFrameRate() );
    this.audioFmt      = audioFmt;
    this.playedSeconds = 0;
    this.playEnabled   = true;
    this.playLock      = new Object();
    setDefaultCloseOperation( JFrame.DO_NOTHING_ON_CLOSE );
    setTitle( "Audiowiedergabe" );
    UIUtil.setIconImagesAt( this );

    // Fensterinhalt
    setLayout( new GridBagLayout() );

    GridBagConstraints gbc = new GridBagConstraints(
					0, 0,
					1, 1,
					0.0, 0.0,
					GridBagConstraints.WEST,
					GridBagConstraints.NONE,
					new Insets( 5, 5, 0, 5 ),
					0, 0 );

    add( new JLabel( "Lautstr\u00E4rke:" ), gbc );

    float volume         = UIUtil.getAudioPlayVolume();
    this.requestedVolume = Float.valueOf( volume );

    this.sliderVolume = new JSlider(
		SwingConstants.HORIZONTAL,
		0,
		VOLUME_SLIDER_MAX,
		Math.round( volume * (float) VOLUME_SLIDER_MAX ) );
    this.sliderVolume.setPaintTicks( true );
    this.sliderVolume.setPaintTrack( true );
    this.sliderVolume.setPaintLabels( false );
    gbc.fill    = GridBagConstraints.HORIZONTAL;
    gbc.weightx = 1.0;
    gbc.gridx++;
    add( this.sliderVolume, gbc );

    this.fldPlayedTime = new JTextField();
    this.fldPlayedTime.setEnabled( false );
    gbc.weightx = 0.0;
    gbc.gridx   = 0;
    gbc.gridy++;
    add( this.fldPlayedTime, gbc );

    this.progressValue = -1;
    this.progressBar   = new JProgressBar(
				SwingConstants.HORIZONTAL,
				this.progressValue,
				PROGRESS_BAR_MAX );
    this.progressBar.setBorderPainted( true );
    gbc.weightx = 1.0;
    gbc.gridx++;
    add( this.progressBar, gbc );

    this.btnClose     = new JButton( UIUtil.ITEM_CLOSE );
    gbc.anchor        = GridBagConstraints.CENTER;
    gbc.insets.top    = 10;
    gbc.insets.bottom = 10;
    gbc.fill          = GridBagConstraints.NONE;
    gbc.weightx       = 0.0;
    gbc.gridwidth     = GridBagConstraints.REMAINDER;
    gbc.gridx         = 0;
    gbc.gridy++;
    add( this.btnClose, gbc );

    // Fensterposition
    setResizable( true );
    if( recentWindowBounds != null ) {
      setBounds( recentWindowBounds );
    } else if( !UIUtil.restoreWindowBounds(
			this,
			Main.getProperties(),
			PROP_PREFIX ) )
    {
      pack();
      setLocationByPlatform( true );
    }

    // Listener
    this.btnClose.addActionListener( this );
    this.sliderVolume.addChangeListener( this );

    // Timer fuer Aktualisierungen
    this.updPlayedTimeTimer = new javax.swing.Timer(
						UPD_PLAYED_TIME_MILLIS,
						this );
    this.updPlayedTimeTimer.start();
    this.updPositionTimer = new javax.swing.Timer(
						UPD_POSITION_MILLIS,
						this );
    this.updPositionTimer.start();

    // Audiowiedergabe starten
    this.playThread = new Thread( this, Main.APPNAME + " Audio Player" );
    this.playThread.start();

    // Sonstiges
    showPlayedSeconds( 0 );
    addWindowListener(
		new WindowAdapter()
		{
		  @Override
		  public void windowClosing( WindowEvent e )
		  {
		    doClose();
		  }
		} );
  }


	/* --- private Methoden --- */

  private void checkRequestedVolume( SourceDataLine line )
  {
    Float requestedVolume = null;
    synchronized( this.playLock ) {
      requestedVolume      = this.requestedVolume;
      this.requestedVolume = null;
    }
    if( requestedVolume != null ) {
      AudioUtil.setVolume( line, requestedVolume.floatValue() );
    }
  }


  private void closeInternal()
  {
    this.btnClose.removeActionListener( this );
    this.sliderVolume.removeChangeListener( this );
    this.updPlayedTimeTimer.stop();
    this.updPositionTimer.stop();
    recentWindowBounds = getBounds();
    setVisible( false );
    dispose();
  }


  private void doClose()
  {
    this.playEnabled = false;
    try {
      this.playThread.interrupt();
      this.playThread.join( 200 );
    }
    catch( InterruptedException ex ) {}
    closeInternal();
  }


  private void playFinished( Exception ex )
  {
    this.analysisFrm.setPlayPosition( -1 );
    if( ex != null ) {
      UIUtil.showErrorMsg( this, ex.getMessage() );
    }
    closeInternal();
  }


  private void showPlayedSeconds( int playedSeconds )
  {
    this.fldPlayedTime.setText(
		String.format(
			"%02d:%02d:%02d",
			playedSeconds / 3600,
			(playedSeconds / 60) % 60,
			playedSeconds % 60 ) );
  }


  private void updPlayedTimeFields()
  {
    if( this.frameRate > 0 ) {
      int playedSeconds = (this.curFrameIdx - this.begFrameIdx)
						/ this.frameRate;
      if( playedSeconds != this.playedSeconds ) {
	this.playedSeconds = playedSeconds;
	showPlayedSeconds( playedSeconds );
      }
    }
    if( this.frameRange > 0 ) {
      int progressValue = 0;
      int curIdx        = this.curFrameIdx - this.begFrameIdx;
      progressValue     = (curIdx * PROGRESS_BAR_MAX) / this.frameRange;
      if( progressValue != this.progressValue ) {
	this.progressValue = progressValue;
	this.progressBar.setValue( progressValue );
      }
    }
  }


  private void updPlayPosition()
  {
    this.analysisFrm.setPlayPosition( this.curFrameIdx );
  }
}
