/*
 * (c) 2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Fenster zum Aneinanderhaengen von mehreren Audiodateien
 */

package jkcload.ui;

import java.awt.BorderLayout;
import java.awt.EventQueue;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.dnd.DnDConstants;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetEvent;
import java.awt.dnd.DropTargetListener;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Properties;
import java.util.Vector;
import javax.sound.sampled.AudioFileFormat;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JScrollPane;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;
import javax.swing.Timer;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.filechooser.FileFilter;
import jkcload.audio.AudioUtil;
import jkcload.audio.ExtByteArrayOutputStream;
import jkcload.Main;


public class AudioFileConcatFrm extends JFrame
				implements
					ActionListener,
					DropTargetListener,
					ListSelectionListener
{
  private class PauseItem
  {
    private int millis;
    private String text;

    private PauseItem( int millis )
    {
      this.millis = millis;
      this.text = formatFloat( (float) millis / 1000F ) + " Sekunden Pause";
    }

    @Override
    public String toString()
    {
      return this.text;
    }
  };

  private static final String PROP_PREFIX
			= Main.PROP_PREFIX + "audio_file_concat.";

  private static final String ITEM_ADD_FILE
			= "Audiodatei hinzuf\u00FCgen...";
  private static final String ITEM_REMOVE = "Entfernen";

  private boolean        notified;
  private Vector<Object> listItems;
  private File           recentOutFile;
  private File           recentSourceDir;
  private Timer          statusTimer;
  private JButton        btnAddFile;
  private JButton        btnRemove;
  private JButton        btnUp;
  private JButton        btnDown;
  private JMenuItem      mnuAddFile;
  private JMenuItem      mnuAddPause;
  private JMenuItem      mnuRemove;
  private JMenuItem      mnuRemoveAll;
  private JMenuItem      mnuSaveAs;
  private JMenuItem      mnuSaveAllAs;
  private JMenuItem      mnuClose;
  private JList<Object>  list;
  private JScrollPane    scrollPane;
  private JLabel         labelStatus;
  private DropTarget     dropTarget1;
  private DropTarget     dropTarget2;


  public AudioFileConcatFrm()
  {
    this.notified = false;
    setDefaultCloseOperation( JFrame.DISPOSE_ON_CLOSE );
    UIUtil.setIconImagesAt( this );
    setTitle( Main.APPNAME + " Audiodateien zusammenf\u00FCgen" );
    this.listItems       = new Vector<>();
    this.recentOutFile   = null;
    this.recentSourceDir = null;


    // Menu
    JMenu mnuFile = new JMenu( "Datei" );
    mnuFile.setMnemonic( KeyEvent.VK_D );

    this.mnuAddFile = new JMenuItem( ITEM_ADD_FILE );
    mnuFile.add( this.mnuAddFile );

    this.mnuAddPause = new JMenuItem( "Pause hinzuf\u00FCgen..." );
    mnuFile.add( this.mnuAddPause );

    this.mnuRemove = new JMenuItem( ITEM_REMOVE );
    this.mnuRemove.setAccelerator(
		KeyStroke.getKeyStroke( KeyEvent.VK_DELETE, 0 ) );
    mnuFile.add( this.mnuRemove );
    mnuFile.addSeparator();

    this.mnuRemoveAll = new JMenuItem( "Alle entfernen" );
    mnuFile.add( this.mnuRemoveAll );
    mnuFile.addSeparator();

    this.mnuSaveAs = new JMenuItem(
			"Markierte in einer Datei speichern unter..." );
    mnuFile.add( this.mnuSaveAs );

    this.mnuSaveAllAs = new JMenuItem(
			"Alle in einer Datei speichern unter..." );
    mnuFile.add( this.mnuSaveAllAs );
    mnuFile.addSeparator();

    this.mnuClose = new JMenuItem( UIUtil.ITEM_CLOSE );
    mnuFile.add( this.mnuClose );

    JMenuBar mnuBar = new JMenuBar();
    mnuBar.add( mnuFile );
    setJMenuBar( mnuBar );


    // Fensterinhalt
    setLayout( new BorderLayout() );

    JToolBar toolBar = new JToolBar();
    add( toolBar, BorderLayout.NORTH );

    this.btnAddFile = UIUtil.createImageButton(
				"/images/open.png",
				ITEM_ADD_FILE );
    toolBar.add( this.btnAddFile );

    this.btnRemove = UIUtil.createImageButton(
				"/images/delete.png",
				ITEM_REMOVE );
    toolBar.add( this.btnRemove );
    toolBar.addSeparator();

    this.btnDown = UIUtil.createImageButton(
				"/images/down.png",
				"Nach unten schieben" );
    toolBar.add( this.btnDown );

    this.btnUp = UIUtil.createImageButton(
				"/images/up.png",
				"Nach oben schieben" );
    toolBar.add( this.btnUp );

    this.list = new JList<>();
    this.list.setSelectionMode(
		ListSelectionModel.MULTIPLE_INTERVAL_SELECTION );
    this.scrollPane = new JScrollPane( this.list );
    add( this.scrollPane, BorderLayout.CENTER );

    this.labelStatus = new JLabel( UIUtil.STATUS_READY );
    this.labelStatus.setBorder(
		BorderFactory.createEmptyBorder( 5, 5, 5, 5 ) );
    add( this.labelStatus, BorderLayout.SOUTH );


    // Drag&Drop
    this.dropTarget1 = new DropTarget( this.list, this );
    this.dropTarget2 = new DropTarget( this.scrollPane, this );


    // Timer zum Zuruecksetzen des Statustextes
    this.statusTimer = new Timer( 5000, this );
    this.statusTimer.setRepeats( false );


    // Sonstiges
    if( !UIUtil.restoreWindowBounds(
			this,
			Main.getProperties(),
			PROP_PREFIX ) )
    {
      setSize( 500, 300 );
      setLocationByPlatform( true );
    }
    updFileActionsEnabled();
    updListActionsEnabled();
    EventQueue.invokeLater( ()->this.btnAddFile.requestFocus() );
  }


  public void getSettings( Properties props )
  {
    UIUtil.getWindowBounds( this, props, PROP_PREFIX );
  }


  public void doClose()
  {
    setVisible( false );
    dispose();
  }


	/* --- ActionListener --- */

  @Override
  public void actionPerformed( ActionEvent e )
  {
    Object src = e.getSource();
    if( (src == this.mnuAddFile) || (src == this.btnAddFile) ) {
      doAddFile();
    } else if( src == this.mnuAddPause ) {
      doAddPause();
    } else if( (src == this.mnuRemove) || (src == this.btnRemove) ) {
      doRemove();
    } else if( src == this.mnuRemoveAll ) {
      doRemoveAll();
    } else if( src == this.mnuSaveAs ) {
      doSaveAs();
    } else if( src == this.mnuSaveAllAs ) {
      saveFile( this.listItems );
    } else if( src == this.mnuClose ) {
      doClose();
    } else if( src == this.btnDown ) {
      doDown();
    } else if( src == this.btnUp ) {
      doUp();
    } else if( src == this.statusTimer ) {
      this.labelStatus.setText( UIUtil.STATUS_READY );
    }
  }


	/* --- DropTargetListener --- */

  @Override
  public void dragEnter( DropTargetDragEvent e )
  {
    UIUtil.acceptDragInCaseOfFileTransfer( e );
  }


  @Override
  public void dragExit( DropTargetEvent e )
  {
    // leer
  }


  @Override
  public void dragOver( DropTargetDragEvent e )
  {
    UIUtil.acceptDragInCaseOfFileTransfer( e );
  }


  @Override
  public void drop( DropTargetDropEvent e )
  {
    e.acceptDrop( DnDConstants.ACTION_COPY );
    try {
      Transferable t = e.getTransferable();
      if( t != null ) {
	if( t.isDataFlavorSupported( DataFlavor.javaFileListFlavor ) ) {
	  Object o = t.getTransferData( DataFlavor.javaFileListFlavor );
	  if( o != null ) {
	    if( o instanceof Collection ) {
	      Collection<?> c = (Collection) o;
	      EventQueue.invokeLater( ()->addFiles( c ) );
	    }
	  }
	}
      }
    }
    catch( Exception ex ) {
      UIUtil.fireShowErrorDnDOperationFailed( this, ex );
    }
  }


  @Override
  public void dropActionChanged( DropTargetDragEvent e )
  {
    // leer
  }


	/* --- ListSelectionListener --- */

  @Override
  public void valueChanged( ListSelectionEvent e )
  {
    if( e.getSource() == this.list )
      EventQueue.invokeLater( ()->updListActionsEnabled() );
  }


	/* --- ueberschriebene Methoden --- */

  @Override
  public void addNotify()
  {
    super.addNotify();
    if( !this.notified ) {
      this.notified = true;
      this.btnAddFile.addActionListener( this );
      this.btnRemove.addActionListener( this );
      this.btnDown.addActionListener( this );
      this.btnUp.addActionListener( this );
      this.mnuAddFile.addActionListener( this );
      this.mnuAddPause.addActionListener( this );
      this.mnuRemove.addActionListener( this );
      this.mnuRemoveAll.addActionListener( this );
      this.mnuSaveAs.addActionListener( this );
      this.mnuSaveAllAs.addActionListener( this );
      this.mnuClose.addActionListener( this );
      this.list.addListSelectionListener( this );
      this.dropTarget1.setActive( true );
      this.dropTarget2.setActive( true );
    }
  }


  @Override
  public void removeNotify()
  {
    if( this.notified ) {
      this.notified = false;
      this.btnAddFile.removeActionListener( this );
      this.btnRemove.removeActionListener( this );
      this.btnDown.removeActionListener( this );
      this.btnUp.removeActionListener( this );
      this.mnuAddFile.removeActionListener( this );
      this.mnuAddPause.removeActionListener( this );
      this.mnuRemove.removeActionListener( this );
      this.mnuRemoveAll.removeActionListener( this );
      this.mnuSaveAs.removeActionListener( this );
      this.mnuSaveAllAs.removeActionListener( this );
      this.mnuClose.removeActionListener( this );
      this.list.removeListSelectionListener( this );
      this.dropTarget1.setActive( false );
      this.dropTarget2.setActive( false );
    }
    super.removeNotify();
  }


	/* --- Aktionen --- */

  private void doAddFile()
  {
    JFileChooser fc = new JFileChooser();
    fc.setDialogTitle( "Audiodateien hinzuf\u00FCgen" );
    fc.setMultiSelectionEnabled( true );
    if( this.recentSourceDir != null ) {
      fc.setCurrentDirectory( this.recentSourceDir );
    }
    FileFilter fileFilter = UIUtil.getAudioFileFilter();
    if( fileFilter != null ) {
      fc.addChoosableFileFilter( fileFilter );
      fc.setFileFilter( fileFilter );
    }
    if( fc.showOpenDialog( this ) == JFileChooser.APPROVE_OPTION ) {
      File[] files = fc.getSelectedFiles();
      if( files != null ) {
	if( files.length > 0 ) {
	  List<File> fileList = new ArrayList<>( files.length );
	  for( File file : files ) {
	    fileList.add( file );
	  }
	  addFiles( fileList );
	  this.recentSourceDir = files[ 0 ].getParentFile();
	}
      }
    }
  }


  private void doAddPause()
  {
    for(;;) {
      String text = JOptionPane.showInputDialog(
			this,
			"Pause in Sekunden:",
			"Pause hinzuf\u00FCgen",
			JOptionPane.QUESTION_MESSAGE );
      if( text == null ) {
	break;
      }
      text = text.trim();

      /*
       * Text mit und ohne Locale-Einstellungen parsen
       * und den kleineren Wert nehmen
       */
      Number value1 = null;
      Number value2 = null;
      try {
	value1 = NumberFormat.getInstance().parse( text );
      }
      catch( ParseException ex ) {}
      try {
	value2 = Float.valueOf( text );
      }
      catch( NumberFormatException ex ) {}
      if( (value1 != null) && (value2 != null) ) {
	if( value1.floatValue() > value2.floatValue() ) {
	  value1 = value2;
	}
      } else if( value1 == null ) {
	value1 = value2;
      }
      if( value1 != null ) {
	int millis = Math.round( value1.floatValue() * 1000F );
	if( (millis > 0) && (millis < 100000) ) {
	  int anchor = this.listItems.size();
	  this.listItems.add( new PauseItem( millis ) );
	  this.list.setListData( this.listItems );
	  this.list.setSelectionInterval( anchor, this.listItems.size() - 1 );
	  updFileActionsEnabled();
	  setStatusText( "Pause hinzugef\u00FCgt" );
	  break;
	}
      }
      UIUtil.showErrorMsg( this, "Ung\u00FCltige Eingabe" );
    }
  }


  private void doRemove()
  {
    int[] indices = this.list.getSelectedIndices();
    if( indices.length > 0 ) {
      int nFiles = 0;
      try {
	Arrays.sort( indices );
	for( int i = indices.length - 1; i >= 0; --i ) {
	  this.listItems.remove( indices[ i ] );
	  nFiles++;
	}
      }
      catch( ArrayIndexOutOfBoundsException ex ) {}
      this.list.setListData( this.listItems );
      updFileActionsEnabled();

      StringBuilder buf = new StringBuilder( 64 );
      appendFileCountTextTo( buf, nFiles );
      buf.append( " entfernt" );
      setStatusText( buf.toString() );
    }
  }


  private void doRemoveAll()
  {
    if( !this.listItems.isEmpty() ) {
      if( JOptionPane.showConfirmDialog(
		this,
		"M\u00F6chten Sie alle Dateien entfernen?",
		UIUtil.TITLE_CONFIRMATIOn,
		JOptionPane.YES_NO_OPTION,
		JOptionPane.WARNING_MESSAGE ) == JOptionPane.YES_OPTION )
      {
	this.listItems.clear();
	this.list.setListData( this.listItems );
	updFileActionsEnabled();
	setStatusText( "Alle Dateien entfernt" );
      }
    }
  }


  private void doSaveAs()
  {
    int[] indices = this.list.getSelectedIndices();
    if( indices.length > 0 ) {
      try {
	Arrays.sort( indices );
	List<Object> items = new ArrayList<>( indices.length );
	for( int i = 0; i < indices.length; i++ ) {
	  items.add( this.listItems.get( indices[ i ] ) );
	}
	saveFile( items );
      }
      catch( ArrayIndexOutOfBoundsException ex ) {}
    }
  }


  private void doDown()
  {
    int[] indices = this.list.getSelectedIndices();
    if( indices.length > 0 ) {
      try {
	Arrays.sort( indices );
	if( indices[ indices.length - 1 ] < (this.listItems.size() - 1) ) {
	  for( int i = indices.length - 1; i >= 0; --i ) {
	    Object obj = this.listItems.remove( indices[ i ] );
	    indices[ i ]++;
	    this.listItems.add( indices[ i ], obj );
	  }
	}
      }
      catch( ArrayIndexOutOfBoundsException ex ) {}
      this.list.setListData( this.listItems );
      fireSetSelectedIndices( indices );
    }
  }


  private void doUp()
  {
    int[] indices = this.list.getSelectedIndices();
    if( indices.length > 0 ) {
      try {
	Arrays.sort( indices );
	if( indices[ 0 ] > 0 ) {
	  for( int i = 0; i < indices.length; i++ ) {
	    Object obj = this.listItems.remove( indices[ i ] );
	    --indices[ i ];
	    this.listItems.add( indices[ i ], obj );
	  }
	}
      }
      catch( ArrayIndexOutOfBoundsException ex ) {}
      this.list.setListData( this.listItems );
      fireSetSelectedIndices( indices );
    }
  }


	/* --- private Methoden --- */

  private void addFiles( Collection c )
  {
    int anchor       = this.listItems.size();
    int nFiles       = 0;
    int totalSeconds = 0;
    try {
      try {
	Iterator iter = c.iterator();
	while( iter.hasNext() ) {
	  Object obj = iter.next();
	  if( obj != null ) {
	    if( obj instanceof File ) {
	      int seconds = getFileLengthInSeconds( (File) obj );
	      if( seconds < 0 ) {
		throw new IOException(
			((File) obj).getPath() + ":\nDatei nicht lesbar"
				+ " oder Format nicht unterst\u00FCtzt" );
	      }
	      totalSeconds += seconds;
	      this.listItems.add( obj );
	      nFiles++;
	    }
	  }
	}
      }
      catch( NoSuchElementException ex ) {}

      if( nFiles > 0 ) {
	this.list.setListData( this.listItems );
	this.list.setSelectionInterval( anchor, this.listItems.size() - 1 );
	updFileActionsEnabled();

	StringBuilder buf = new StringBuilder( 64 );
	appendFileCountTextTo( buf, nFiles );
	if( totalSeconds > 0 ) {
	  String s = getTimeTextBySeconds( totalSeconds );
	  if( s != null ) {
	    buf.append( " mit " );
	    if( nFiles > 1 ) {
	      buf.append( "zusammen " );
	    }
	    buf.append( s );
	    buf.append( " L\u00E4nge" );
	  }
	}
	buf.append( " hinzugef\u00FCgt" );
	setStatusText( buf.toString() );
      }
    }
    catch( IOException ex ) {
      UIUtil.showErrorMsg( this, null, ex );
    }
  }


  private static void appendFileCountTextTo( StringBuilder buf, int n )
  {
    if( n == 1 ) {
      buf.append( "Datei" );
    } else {
      buf.append( n );
      buf.append( " Dateien" );
    }
  }


  private static void appendPauseTo(
			OutputStream out,
			AudioFormat  audioFmt,
			int          millis ) throws IOException
  {
    int nFrames = Math.round(
			audioFmt.getFrameRate() * (float) millis / 1000F );
    int nBytes  = nFrames * audioFmt.getFrameSize();
    for( int i = 0; i < nBytes; i++ ) {
      out.write( 0 );
    }
  }


  private void fireSetSelectedIndices( int[] indices )
  {
    EventQueue.invokeLater( ()->setSelectedIndices( indices ) );
  }


  private static String formatFloat( float value )
  {
    NumberFormat numFmt = NumberFormat.getInstance();
    if( numFmt instanceof DecimalFormat ) {
      ((DecimalFormat) numFmt).applyPattern( "#0.0##" );
    }
    return numFmt.format( value );
  }


  private static int getFileLengthInSeconds( File file )
  {
    int rv = -1;
    if( file != null ) {
      try( AudioInputStream in = AudioSystem.getAudioInputStream( file ) ) {
	rv = getSeconds( in.getFormat(), in.getFrameLength() );
      }
      catch( Exception ex ) {}
    }
    return rv;
  }


  private static int getSeconds( AudioFormat fmt, long len )
  {
    int rv = -1;
    if( (fmt != null) && (len >= 0) ) {
      float frameRate = fmt.getFrameRate();
      if( frameRate >= 0F ) {
	int millis = Math.round( (float) len / frameRate * 1000F );
	rv         = (millis + 500) / 1000;
      }
    }
    return rv;
  }


  private static String getTimeTextBySeconds( int seconds )
  {
    String rv = null;
    if( seconds >= 0 ) {
      int minutes = seconds / 60;
      int hours   = minutes / 60;
      if( hours > 0 ) {
	rv = String.format(
			"%1d:%02d:%02d",
			hours,
			minutes % 60,
			seconds % 60 );
      } else {
	if( minutes > 0 ) {
	  rv = String.format(
			"%02d:%02d Minuten",
			minutes,
			seconds % 60 );
	} else {
	  if( seconds == 1 ) {
	    rv = "1 Sekunde";
	  } else {
	    rv = String.format( "%1d Sekunden", seconds );
	  }
	}
      }
    }
    return rv;
  }


  private void saveFile( List<Object> items )
  {
    boolean hasFiles = false;
    for( Object item : items ) {
      if( item instanceof File ) {
	hasFiles = true;
	break;
      }
    }
    if( hasFiles ) {
      File outFile = UIUtil.showSaveFileDlg(
				this,
				"Audiodatei speichern",
				this.recentOutFile,
				null,
				UIUtil.getAudioFileFilter(),
				null );
      if( outFile != null ) {
	try {
	  AudioFileFormat.Type     fileType = UIUtil.getAudioFileType(
							outFile.getName() );
	  byte[]                   readBuf  = new byte[ 0x1000 ];
	  AudioFormat              audioFmt = null;
	  ExtByteArrayOutputStream audioBuf = new ExtByteArrayOutputStream(
								0x4000 );

	  // Dateien lesen
	  int pauseMillis = 0;
	  for( Object item : items ) {
	    if( item instanceof PauseItem ) {
	      int millis = ((PauseItem) item).millis;
	      if( audioFmt != null ) {
		appendPauseTo( audioBuf, audioFmt, millis );
	      } else {
		pauseMillis += millis;
	      }
	    } else if( item instanceof File ) {
	      AudioInputStream in = null;
	      try {
		in = AudioSystem.getAudioInputStream( (File) item );
		AudioFormat fmt = in.getFormat();
		if( audioFmt != null ) {
		  try {
		    in = AudioSystem.getAudioInputStream( audioFmt, in );
		  }
		  catch( IllegalArgumentException ex ) {
		    throw new IOException(
			((File) item).getPath()
				+ ":\nDas Audioformat kann nicht"
				+ " in das Format\n"
				+ "der zu speichernden Datei konvertiert"
				+ " werden." );
		  }
		} else {
		  audioFmt = fmt;
		  if( pauseMillis > 0 ) {
		    appendPauseTo( audioBuf, audioFmt, pauseMillis );
		    pauseMillis= 0;
		  }
		}
		int nRead = in.read( readBuf );
		while( nRead > 0 ) {
		  audioBuf.write( readBuf );
		  nRead = in.read( readBuf );
		}
	      }
	      finally {
		AudioUtil.closeSilently( in );
	      }
	    }
	  }

	  // Audiodaten speichern
	  if( audioFmt == null ) {
	    throw new IOException( "Unbekanntes Audioformat" );
	  }
	  int frameSize = audioFmt.getFrameSize();
	  if( frameSize < 1 ) {
	    throw new IOException( "Ung\u00FCltiges Audioformat" );
	  }
	  long nFrames = audioBuf.size() / frameSize;
	  if( nFrames <= 0 ) {
	    throw new IOException( "Keine Audiodaten vorhanden" );
	  }
	  OutputStream out = null;
	  try {
	    out = new BufferedOutputStream(
				new FileOutputStream( outFile ) );
	    AudioSystem.write(
			new AudioInputStream(
				audioBuf.createInputStream(),
				audioFmt,
				nFrames ),
			fileType,
			out );
	    out.close();
	    out                = null;
	    this.recentOutFile = outFile;
	  }
	  finally {
	    AudioUtil.closeSilently( out );
	  }
	  StringBuilder buf = new StringBuilder( 64 );
	  buf.append( "Datei" );
	  int seconds = getSeconds( audioFmt, nFrames );
	  if( seconds > 0 ) {
	    buf.append( " mit " );
	    buf.append( getTimeTextBySeconds( seconds ) );
	    buf.append( " L\u00E4nge" );
	  }
	  buf.append( " gespeichert" );
	  setStatusText( buf.toString() );
	}
	catch( Exception ex ) {
	  UIUtil.showErrorMsg(
		this,
		"Audiodaten konnten nicht gespeichert werden.",
		ex );
	}
      }
    } else {
      UIUtil.showErrorMsg(
		this,
		"Die zu speichernden Eintr\u00E4ge enthllten"
			+ " keine einzige Audiodatei." );
    }
  }


  private void setSelectedIndices( int[] indices )
  {
    this.list.clearSelection();
    for( int idx : indices ) {
      this.list.addSelectionInterval( idx, idx );
    }
  }


  private void setStatusText( String text )
  {
    this.labelStatus.setText( text );
    this.statusTimer.restart();
  }


  private void updFileActionsEnabled()
  {
    boolean state = !this.listItems.isEmpty();
    this.mnuRemoveAll.setEnabled( state );
    this.mnuSaveAllAs.setEnabled( state );
  }


  private void updListActionsEnabled()
  {
    int[] indices = this.list.getSelectedIndices();
    Arrays.sort( indices );

    boolean state = (indices.length > 0);
    this.btnRemove.setEnabled( state );
    this.mnuRemove.setEnabled( state );
    this.mnuSaveAs.setEnabled( state );

    state = false;
    if( indices.length > 0 ) {
      if( indices[ indices.length - 1 ] < (this.listItems.size() - 1) ) {
	state = true;
      }
    }
    this.btnDown.setEnabled( state );

    state = false;
    if( indices.length > 0 ) {
      if( indices[ 0 ] > 0 ) {
	state = true;
      }
    }
    this.btnUp.setEnabled( state );
  }
}
