/*
 * (c) 2023-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Fenster zur Anzeige der Analysedaten
 */

package jkcload.ui;

import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.InputStream;
import java.io.IOException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.util.Properties;
import java.util.Set;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import javax.sound.sampled.AudioFileFormat;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileNameExtensionFilter;
import jkcload.Main;
import jkcload.audio.AudioProcessor;
import jkcload.audio.AudioUtil;


public class AnalysisFrm extends JFrame implements ActionListener
{
  private static final String ANALYSIS_FILE_EXTENSION = "jkc";
  private static final String ANALYSIS_FILE_HEADER    = "JKCL:ANALYSIS";
  private static final String LABEL_FROM_TO = "Markiert von/bis:";
  private static final String PROP_PREFIX   = Main.PROP_PREFIX + "analysis.";

  private boolean     notified;
  private int         cursorIdx;
  private int         fromToIdx;
  private AudioFormat audioFmt;
  private byte[]      audioData;
  private byte[]      recognitionData;
  private FileFilter  analysisFileFilter;
  private FileFilter  audioFileFilter;
  private File        curFile;
  private File        recentAnalysisDir;
  private File        recentAudioDataDir;
  private File        recentPhaseDataDir;
  private JMenuItem   mnuAnalysisOpen;
  private JMenuItem   mnuAnalysisSaveAs;
  private JMenuItem   mnuAudioDataSaveAs;
  private JMenuItem   mnuSelectedAudioDataSaveAs;
  private JMenuItem   mnuPhaseDataSaveAs;
  private JMenuItem   mnuAudioDataInfo;
  private JMenuItem   mnuAudioDataPlay;
  private JMenuItem   mnuSelectedAudioDataPlay;
  private JMenuItem   mnuClose;
  private JMenuItem   mnuGotoFramePos;
  private JMenuItem   mnuGotoTimestamp;
  private JMenuItem   mnuMarkCurPos;
  private JMenuItem   mnuUnmark;
  private JButton     btnScalePlus;
  private JButton     btnScaleMinus;
  private JLabel      labelCursorFrameIdx;
  private JLabel      labelCursorTime;
  private JLabel      labelFromToFrameIdx;
  private JLabel      labelFromToTime;
  private JLabel      labelNoContent;
  private JTextField  fldCursorFrameIdx;
  private JTextField  fldCursorTime;
  private JTextField  fldFromToFrameIdx;
  private JTextField  fldFromToTime;
  private JScrollPane scrollPane;
  private AnalysisFld analysisFld;


  public AnalysisFrm()
  {
    this.notified           = false;
    this.cursorIdx          = -1;
    this.fromToIdx          = -1;
    this.audioFmt           = null;
    this.audioData          = null;
    this.recognitionData    = null;
    this.curFile            = null;
    this.recentAnalysisDir  = null;
    this.recentAudioDataDir = null;
    this.recentPhaseDataDir = null;
    setDefaultCloseOperation( JFrame.DO_NOTHING_ON_CLOSE );
    updTitle();
    UIUtil.setIconImagesAt( this );

    this.audioFileFilter    = UIUtil.getAudioFileFilter();
    this.analysisFileFilter = new FileNameExtensionFilter(
				String.format(
					"%s Analysedateien (*.%s)",
					Main.APPNAME,
					ANALYSIS_FILE_EXTENSION ),
				ANALYSIS_FILE_EXTENSION );


    // Menu
    JMenu mnuFile = new JMenu( "Datei" );
    mnuFile.setMnemonic( KeyEvent.VK_D );

    this.mnuAnalysisOpen = new JMenuItem( "Analysedatei \u00F6ffnen..." );
    mnuFile.add( this.mnuAnalysisOpen );

    this.mnuAnalysisSaveAs = new JMenuItem(
				"Analysedaten speichern unter..." );
    this.mnuAnalysisSaveAs.setEnabled( false );
    mnuFile.add( this.mnuAnalysisSaveAs );
    mnuFile.addSeparator();

    if( this.audioFileFilter != null ) {
      this.mnuAudioDataSaveAs = new JMenuItem(
			"Gelesene Audiodaten speichern unter..." );
      this.mnuAudioDataSaveAs.setEnabled( false );
      mnuFile.add( this.mnuAudioDataSaveAs );

      this.mnuSelectedAudioDataSaveAs = new JMenuItem(
			"Markierte Audiodaten speichern unter..." );
      this.mnuSelectedAudioDataSaveAs.setEnabled( false );
      mnuFile.add( this.mnuSelectedAudioDataSaveAs );

      this.mnuPhaseDataSaveAs = new JMenuItem(
			"Ermittelte Phasenlagedaten speichern unter..." );
      this.mnuPhaseDataSaveAs.setEnabled( false );
      mnuFile.add( this.mnuPhaseDataSaveAs );
    } else {
      this.mnuAudioDataSaveAs         = null;
      this.mnuSelectedAudioDataSaveAs = null;
      this.mnuPhaseDataSaveAs         = null;
    }
    mnuFile.addSeparator();

    this.mnuAudioDataInfo = new JMenuItem(
			"\u00DCber gelesene Audiodaten..." );
    this.mnuAudioDataInfo.setEnabled( false );
    mnuFile.add( this.mnuAudioDataInfo );

    this.mnuAudioDataPlay = new JMenuItem(
			"Gelesene Audiodaten abspielen..." );
    this.mnuAudioDataPlay.setEnabled( false );
    mnuFile.add( this.mnuAudioDataPlay );

    this.mnuSelectedAudioDataPlay = new JMenuItem(
			"Markierte Audiodaten abspielen..." );
    this.mnuSelectedAudioDataPlay.setEnabled( false );
    mnuFile.add( this.mnuSelectedAudioDataPlay );
    mnuFile.addSeparator();

    this.mnuClose = new JMenuItem( UIUtil.ITEM_CLOSE );
    mnuFile.add( this.mnuClose );

    JMenu mnuEdit = new JMenu( "Bearbeiten" );
    mnuEdit.setMnemonic( KeyEvent.VK_B );

    this.mnuMarkCurPos = new JMenuItem(
				"Aktuelle Position als von/bis setzen" );
    this.mnuMarkCurPos.setEnabled( false );
    mnuEdit.add( this.mnuMarkCurPos );

    this.mnuUnmark = new JMenuItem( "Auswahl aufheben" );
    this.mnuUnmark.setEnabled( false );
    mnuEdit.add( this.mnuUnmark );
    mnuEdit.addSeparator();

    this.mnuGotoFramePos = new JMenuItem( "Gehe zu Frame-Position..." );
    this.mnuGotoFramePos.setEnabled( false );
    mnuEdit.add( this.mnuGotoFramePos );

    this.mnuGotoTimestamp = new JMenuItem( "Gehe zu Zeitpunkt..." );
    this.mnuGotoTimestamp.setEnabled( false );
    mnuEdit.add( this.mnuGotoTimestamp );

    JMenuBar mnuBar = new JMenuBar();
    mnuBar.add( mnuFile );
    mnuBar.add( mnuEdit );
    setJMenuBar( mnuBar );


    // Fensterinhalt
    setLayout( new GridBagLayout() );

    GridBagConstraints gbc = new GridBagConstraints(
					0, 0,
					GridBagConstraints.REMAINDER, 1,
					1.0, 1.0,
					GridBagConstraints.WEST,
					GridBagConstraints.BOTH,
					new Insets( 5, 5, 0, 5 ),
					0, 0 );

    this.analysisFld = new AnalysisFld( this );
    this.scrollPane  = new JScrollPane( this.analysisFld );
    this.scrollPane.getViewport().setBackground( AnalysisFld.BG_COLOR );
    this.scrollPane.setWheelScrollingEnabled( false );
    this.scrollPane.setRowHeaderView(
			this.analysisFld.createRowHeaderView() );
    add( this.scrollPane, gbc );

    JPanel panelScale = new JPanel( new GridLayout( 1, 2, 5, 5 ) );
    gbc.fill          = GridBagConstraints.NONE;
    gbc.weightx       = 0.0;
    gbc.weighty       = 0.0;
    gbc.gridwidth     = 1;
    gbc.gridy++;
    add( panelScale, gbc );

    this.btnScaleMinus = new JButton( "-" );
    this.btnScaleMinus.setToolTipText( "Verkleinern" );
    panelScale.add( this.btnScaleMinus );

    this.btnScalePlus = new JButton( "+" );
    this.btnScalePlus.setToolTipText( "Vergr\u00F6\u00DFern" );
    panelScale.add( this.btnScalePlus );

    Font font = this.btnScalePlus.getFont();
    if( font != null ) {
      font = new Font( font.getName(), Font.BOLD, font.getSize() );
      this.btnScalePlus.setFont( font );
      this.btnScaleMinus.setFont( font );
    }

    this.labelCursorFrameIdx = new JLabel( "Aktuelle Frame-Position:" );
    gbc.insets.left          = 20;
    gbc.gridx++;
    add( this.labelCursorFrameIdx, gbc );

    this.fldCursorFrameIdx = new JTextField( 10 );
    this.fldCursorFrameIdx.setEditable( false );
    gbc.insets.left = 5;
    gbc.fill        = GridBagConstraints.HORIZONTAL;
    gbc.weightx     = 0.5;
    gbc.gridx++;
    add( this.fldCursorFrameIdx, gbc );

    this.labelCursorTime = new JLabel( "Aktueller Zeitpunkt:" );
    gbc.anchor           = GridBagConstraints.EAST;
    gbc.insets.left      = 20;
    gbc.fill             = GridBagConstraints.NONE;
    gbc.weightx          = 0.0;
    gbc.gridx++;
    add( this.labelCursorTime, gbc );

    this.fldCursorTime = new JTextField( 10 );
    this.fldCursorTime.setEditable( false );
    gbc.anchor      = GridBagConstraints.WEST;
    gbc.insets.left = 5;
    gbc.fill        = GridBagConstraints.HORIZONTAL;
    gbc.weightx     = 0.5;
    gbc.gridx++;
    add( this.fldCursorTime, gbc );

    this.labelFromToFrameIdx = new JLabel( LABEL_FROM_TO );
    gbc.anchor               = GridBagConstraints.EAST;
    gbc.insets.bottom        = 5;
    gbc.insets.left          = 20;
    gbc.fill                 = GridBagConstraints.NONE;
    gbc.weightx              = 0.0;
    gbc.gridx                = 1;
    gbc.gridy++;
    add( this.labelFromToFrameIdx, gbc );

    this.fldFromToFrameIdx = new JTextField( 10 );
    this.fldFromToFrameIdx.setEditable( false );
    gbc.anchor      = GridBagConstraints.WEST;
    gbc.insets.left = 5;
    gbc.fill        = GridBagConstraints.HORIZONTAL;
    gbc.weightx     = 0.5;
    gbc.gridx++;
    add( this.fldFromToFrameIdx, gbc );

    this.labelFromToTime = new JLabel( LABEL_FROM_TO );
    gbc.anchor           = GridBagConstraints.EAST;
    gbc.insets.left      = 20;
    gbc.fill             = GridBagConstraints.NONE;
    gbc.weightx          = 0.0;
    gbc.gridx++;
    add( this.labelFromToTime, gbc );

    this.fldFromToTime = new JTextField( 10 );
    this.fldFromToTime.setEditable( false );
    gbc.anchor      = GridBagConstraints.WEST;
    gbc.insets.left = 5;
    gbc.fill        = GridBagConstraints.HORIZONTAL;
    gbc.weightx     = 0.5;
    gbc.gridx++;
    add( this.fldFromToTime, gbc );


    // Sonstiges
    updFileFieldsEnabled();
    updScaleFieldsEnabled();
    if( !UIUtil.restoreWindowBounds(
			this,
			Main.getProperties(),
			PROP_PREFIX ) )
    {
      pack();
      setLocationByPlatform( true );
    }
    this.fldCursorFrameIdx.setColumns( 0 );
    this.fldCursorTime.setColumns( 0 );
    this.fldFromToFrameIdx.setColumns( 0 );
    this.fldFromToTime.setColumns( 0 );
    addWindowListener(
		new WindowAdapter()
		{
		  @Override
		  public void windowClosing( WindowEvent e )
		  {
		    doClose();
		  }
		} );
    EventQueue.invokeLater( ()->this.btnScalePlus.requestFocus() );
  }


  public void cursorIdxChanged( int cursorIdx, int markIdx )
  {
    EventQueue.invokeLater(
		()->cursorIdxChangedInternal( cursorIdx, markIdx ) );
  }


  public void doClose()
  {
    AudioPlayFrm.close();
    setVisible( false );
    dispose();
  }


  public void getSettings( Properties props )
  {
    UIUtil.getWindowBounds( this, props, PROP_PREFIX );
  }


  public void setAnalysisData(
			File        audioFile,
			AudioFormat audioFmt,
			byte[]      audioData,
			byte[]      recognitionData )
  {
    this.curFile         = audioFile;
    this.audioFmt        = audioFmt;
    this.audioData       = audioData;
    this.recognitionData = recognitionData;
    this.analysisFld.setAnalysisData( audioFmt, audioData, recognitionData );
    this.scrollPane.revalidate();
    updScaleFieldsEnabled();
    updFileFieldsEnabled();
    updTitle();
  }


  public void setPlayPosition( int idx )
  {
    this.analysisFld.setPlayPosition( idx );
  }


	/* --- ActionListener --- */

  @Override
  public void actionPerformed( ActionEvent e )
  {
    Object src = e.getSource();
    if( src == this.mnuAnalysisOpen ) {
      doAnalysisOpen();
    } else if( src == this.mnuAnalysisSaveAs ) {
      doAnalysisSaveAs();
    } else if( src == this.mnuAudioDataSaveAs ) {
      doAudioDataSaveAs();
    } else if( src == this.mnuSelectedAudioDataSaveAs ) {
      doSelectedAudioDataSaveAs();
    } else if( src == this.mnuPhaseDataSaveAs ) {
      doPhaseDataSaveAs();
    } else if( src == this.mnuAudioDataInfo ) {
      doAudioDataInfo();
    } else if( src == this.mnuAudioDataPlay ) {
      doAudioDataPlay();
    } else if( src == this.mnuSelectedAudioDataPlay ) {
      doSelectedAudioDataPlay();
    } else if( src == this.mnuClose ) {
      setVisible( false );
    } else if( src == this.mnuGotoFramePos ) {
      doGotoFramePos();
    } else if( src == this.mnuGotoTimestamp ) {
      doGotoTimestamp();
    } else if( src == this.mnuMarkCurPos ) {
      if( this.cursorIdx >= 0 ) {
	this.analysisFld.setFromTo( this.cursorIdx );
      }
    } else if( src == this.mnuUnmark ) {
      this.analysisFld.unmark();
    } else if( src == this.btnScalePlus ) {
      this.analysisFld.scalePlus();
      updScaleFieldsEnabled();
    } else if( src == this.btnScaleMinus ) {
      this.analysisFld.scaleMinus();
      updScaleFieldsEnabled();
    }
  }


	/* --- ueberschriebene Methoden --- */

  @Override
  public void addNotify()
  {
    super.addNotify();
    if( !this.notified ) {
      this.notified = true;
      this.mnuAnalysisOpen.addActionListener( this );
      this.mnuAnalysisSaveAs.addActionListener( this );
      this.mnuAudioDataInfo.addActionListener( this );
      this.mnuAudioDataPlay.addActionListener( this );
      this.mnuAudioDataSaveAs.addActionListener( this );
      this.mnuSelectedAudioDataPlay.addActionListener( this );
      this.mnuSelectedAudioDataSaveAs.addActionListener( this );
      this.mnuPhaseDataSaveAs.addActionListener( this );
      this.mnuClose.addActionListener( this );
      this.mnuGotoFramePos.addActionListener( this );
      this.mnuGotoTimestamp.addActionListener( this );
      this.mnuMarkCurPos.addActionListener( this );
      this.mnuUnmark.addActionListener( this );
      this.btnScalePlus.addActionListener( this );
      this.btnScaleMinus.addActionListener( this );
    }
  }


  @Override
  public void removeNotify()
  {
    if( this.notified ) {
      this.notified = false;
      this.mnuAnalysisOpen.removeActionListener( this );
      this.mnuAnalysisSaveAs.removeActionListener( this );
      this.mnuAudioDataInfo.removeActionListener( this );
      this.mnuAudioDataPlay.removeActionListener( this );
      this.mnuAudioDataSaveAs.removeActionListener( this );
      this.mnuSelectedAudioDataPlay.removeActionListener( this );
      this.mnuSelectedAudioDataSaveAs.removeActionListener( this );
      this.mnuPhaseDataSaveAs.removeActionListener( this );
      this.mnuClose.removeActionListener( this );
      this.mnuGotoFramePos.removeActionListener( this );
      this.mnuGotoTimestamp.removeActionListener( this );
      this.mnuMarkCurPos.removeActionListener( this );
      this.mnuUnmark.removeActionListener( this );
      this.btnScalePlus.removeActionListener( this );
      this.btnScaleMinus.removeActionListener( this );
    }
    super.removeNotify();
  }


  @Override
  public void setVisible( boolean state )
  {
    super.setVisible( state );
    if( !state ) {
      clear();
    }
  }


	/* --- Aktionen --- */

  private void doAnalysisOpen()
  {
    File file = UIUtil.showOpenFileDlg(
				this,
				"Analysedatei \u00F6ffnen",
				this.recentAnalysisDir,
				this.analysisFileFilter );
    if( file != null ) {
      try {
	InputStream in = null;
	try {
	  in = new BufferedInputStream( new FileInputStream( file ) );

	  // Kennung pruefen
	  int n = ANALYSIS_FILE_HEADER.length();
	  for( int i = 0; i < n; i++ ) {
	    if( in.read() != ANALYSIS_FILE_HEADER.charAt( i ) ) {
	      throw new IOException(
			String.format(
				"Keine %s Analysedatei",
				Main.APPNAME ) );
	    }
	  }
	  int delimByte = in.read();
	  int version   = in.read();
	  int flags     = in.read();

	  DataInputStream dIn = null;
	  if( (flags & 0x80) != 0 ) {
	    dIn = new DataInputStream( new GZIPInputStream( in ) );
	  } else {
	    dIn = new DataInputStream( in );
	  }
	  in = dIn;

	  int     frameCount        = dIn.readInt();
	  int     frameRate         = dIn.readInt();
	  int     sampleSizeInBits  = dIn.readByte() & 0xFF;
	  int     channels          = dIn.readByte() & 0xFF;
	  boolean dataSigned        = dIn.readBoolean();
	  boolean bigEndian         = dIn.readBoolean();
	  int     frameSize         = dIn.readInt();
	  int     sampleSizeInBytes = (sampleSizeInBits + 7) / 8;
	  if( (delimByte != 0)
	      || (frameCount < 1)
	      || (frameRate < 1)
	      || (sampleSizeInBits < 1)
	      || (channels < 1)
	      || (frameSize != (sampleSizeInBytes * channels)) )
	  {
	    throw new IOException(
			"Datei enth\u00E4lt mysteri\u00F6se Daten." );
	  }
	  if( (version & 0xF0) != 0x10 ) {
	    throw new IOException(
			"Datei enth\u00E4lt Analysedaten in einer"
				+ " nicht unterst\u00FCtzten Version." );
	  }

	  byte[] audioData = new byte[ frameCount * frameSize ];
	  dIn.readFully( audioData );

	  byte[] recognitionData = new byte[ frameCount ];
	  dIn.readFully( recognitionData );

	  dIn.close();
	  in = null;
	  this.recentAnalysisDir = file.getParentFile();
	  setAnalysisData(
			file,
			new AudioFormat(
				(float) frameRate,
				sampleSizeInBits,
				channels,
				dataSigned,
				bigEndian ),
			audioData,
			recognitionData );
	}
	finally {
	  AudioUtil.closeSilently( in );
	}
      }
      catch( EOFException ex ) {
	UIUtil.showErrorMsg( this, "Unerwartetes Dateiende" );
      }
      catch( IOException ex ) {
	UIUtil.showErrorMsg( this, ex.getMessage() );
      }
    }
  }


  private void doAnalysisSaveAs()
  {
    if( (this.audioFmt != null)
	&& (this.audioData != null)
	&& (this.recognitionData != null) )
    {
      boolean              dataSigned = false;
      boolean              pcm        = false;
      AudioFormat.Encoding encoding   = this.audioFmt.getEncoding();
      if( encoding.equals( AudioFormat.Encoding.PCM_UNSIGNED ) ) {
	dataSigned = false;
	pcm        = true;
      } else if( encoding.equals( AudioFormat.Encoding.PCM_SIGNED ) ) {
	dataSigned = true;
	pcm        = true;
      }
      int sampleSizeInBits = this.audioFmt.getSampleSizeInBits();
      int channels         = this.audioFmt.getChannels();
      int frameRate        = Math.round( this.audioFmt.getFrameRate() );
      int frameSize        = ((sampleSizeInBits + 7) / 8) * channels;

      if( pcm
	  && (sampleSizeInBits > 0) && (channels > 0)
	  && (frameRate > 0) && (frameSize > 0) )
      {
	int frameCount = this.audioData.length / frameSize;
	if( frameCount > 0 ) {
	  File file = UIUtil.showSaveFileDlg(
		this,
		"Analysedaten speichern",
		this.recentAnalysisDir,
		null,
		this.analysisFileFilter,
		null );
	  if( file != null ) {
	    try {
	      OutputStream out = null;
	      try {
		out = new BufferedOutputStream(
				new FileOutputStream( file ) );
		out.write( ANALYSIS_FILE_HEADER.getBytes( "US-ASCII" ) );
		out.write( 0 );		// String-Terminierung
		out.write( 0x10 );	// Dateiformat Version 1.0
		out.write( 0x80 );	// nachfolgend GZIP-komprimiert

		DataOutputStream dOut = new DataOutputStream(
				new GZIPOutputStream( out ) );
		out = dOut;
		dOut.writeInt( frameCount );
		dOut.writeInt( frameRate );
		dOut.writeByte( sampleSizeInBits );
		dOut.writeByte( channels );
		dOut.writeBoolean( dataSigned );
		dOut.writeBoolean( this.audioFmt.isBigEndian() );
		dOut.writeInt( frameSize );
		dOut.write( this.audioData, 0, frameCount * frameSize );
		int n = Math.min( frameCount, this.recognitionData.length );
		out.write( this.recognitionData, 0, n );
		while( n < frameCount ) {
		  dOut.write( 0 );
		  n++;
		}
		dOut.close();
		out = null;
		this.recentAnalysisDir = file.getParentFile();
	      }
	      finally {
		AudioUtil.closeSilently( out );
	      }
	    }
	    catch( IOException ex ) {
	      UIUtil.showErrorMsg( this, ex.getMessage() );
	    }
	  }
	}
      }
    }
  }


  private void doAudioDataInfo()
  {
    if( (this.audioFmt != null) && (this.audioData != null) ) {
      StringBuilder buf       = new StringBuilder( 256 );
      float         frameRate = this.audioFmt.getFrameRate();
      if( frameRate >= 1F ) {
	buf.append( Math.round( frameRate ) );
	buf.append( " Hz, " );
      }
      int sampleSizeInBits = this.audioFmt.getSampleSizeInBits();
      buf.append( sampleSizeInBits );
      buf.append( " Bit" );
      int channels = this.audioFmt.getChannels();
      switch( channels ) {
	case 1:		// Sie gelesenen Audiodaten sollten immer Mono sein.
	  buf.append( " Mono" );
	  break;
	case 2:
	  buf.append( " Stereo" );
	  break;
	default:
	  buf.append( ", " );
	  buf.append( channels );
	  buf.append( " Kan\u00E4le" );
      }
      int sampleSizeInBytes = (sampleSizeInBits + 7) / 8;
      if( (frameRate >= 1F) && (sampleSizeInBytes > 0) ) {
	int ms = Math.round( (float) this.audioData.length
					* 1000F
					/ (float) sampleSizeInBytes
					/ frameRate );
	if( ms > 0 ) {
	  buf.append( "\nL\u00E4nge: " );
	  boolean done = false;
	  if( ms >= 60000 ) {
	    int msUp    = ms + 500;		// auf Sekunden runden
	    int hours   = msUp / 1000 / 60 / 60;
	    int minutes = (msUp / 1000 / 60) % 60;
	    int seconds = (msUp / 1000) % 60;
	    if( hours > 0 ) {
	      buf.append( String.format(
				"%d:%02d:%02d Stunden",
				hours,
				minutes,
				seconds ) );
	      done = true;
	    } else if( minutes > 0 ) {
	      buf.append( String.format(
				"%d:%02d Minuten",
				minutes,
				seconds ) );
	      done = true;
	    }
	  }
	  if( !done ) {
	    NumberFormat numFmt = NumberFormat.getInstance();
	    if( numFmt instanceof DecimalFormat ) {
	      ((DecimalFormat) numFmt).applyPattern( "#0.0##" );
	    }
	    buf.append( numFmt.format( (float) ms / 1000F ) );
	    buf.append( " Sekunden" );
	  }
	}
      }
      JOptionPane.showMessageDialog(
		this,
		buf.toString(),
		"\u00DCber gelesene Audiodaten",
		JOptionPane.INFORMATION_MESSAGE );
    }
  }


  private void doAudioDataPlay()
  {
    if( (this.audioFmt != null) && (this.audioData != null) ) {
      playAudio(
		this.audioFmt,
		this.audioData,
		0,
		this.audioData.length );
    }
  }


  private void doAudioDataSaveAs()
  {
    if( (this.audioFmt != null) && (this.audioData != null) ) {
      File file = saveAudioFileAs(
			this.audioFmt,
			this.audioData,
			0,
			this.audioData.length,
			"Eingelesene Audiodaten speichern",
			this.recentAudioDataDir );
      if( file != null ) {
	this.recentAudioDataDir = file.getParentFile();
      }
    }
  }


  private void doGotoFramePos()
  {
    if( this.recognitionData != null ) {
      if( this.recognitionData.length > 0 ) {
	String text = null;
	int    idx  = -1;
	for(;;) {
	  text = JOptionPane.showInputDialog(
			this,
			"Frame-Position:",
			"Zu Frame-Position gehen",
			JOptionPane.QUESTION_MESSAGE );
	  if( text == null ) {
	    break;
	  }
	  try {
	    idx = Integer.parseInt( text );
	    if( idx >= this.recognitionData.length ) {
	      idx = this.recognitionData.length - 1;
	    }
	  }
	  catch( NumberFormatException ex ) {
	    idx = -1;
	  }
	  if( idx > 0 ) {
	    this.analysisFld.setCursorIdx( idx );
	    scrollToIdx( idx );
	    break;
	  } else {
	    UIUtil.showErrorMsg(
		this,
		"Ung\u00FCltige Eingabe!\n"
			+ "Geben Sie bitte eine ganze Zahl von 0 bis "
			+ String.valueOf( this.recognitionData.length - 1 )
			+ " ein!" );
	  }
	}
      }
    }
  }


  private void doGotoTimestamp()
  {
    if( (this.audioFmt != null) && (this.recognitionData != null) ) {
      float frameRate = this.audioFmt.getFrameRate();
      if( (frameRate >= 1F) && (this.recognitionData.length > 0) ) {
	String text = null;
	int    idx  = -1;
	for(;;) {
	  text = JOptionPane.showInputDialog(
			this,
			"Zeitpunkt:",
			"Zu Zeitpunkt gehen",
			JOptionPane.QUESTION_MESSAGE );
	  if( text == null ) {
	    break;
	  }
	  try {
	    String   hText = "0";
	    String   mText = "0";
	    String   sText = "0";
	    String[] elems = text.split( ":" );
	    switch( elems.length ) {
	      case 1:
		sText = elems[ 0 ];
		break;
	      case 2:
		mText = elems[ 0 ];
		sText = elems[ 1 ];
		break;
	      case 3:
		hText = elems[ 0 ];
		mText = elems[ 1 ];
		sText = elems[ 2 ];
		break;
	    }
	    int   hour   = Integer.parseInt( hText );
	    int   minute = Integer.parseInt( mText );
	    float second = Float.parseFloat( sText.replace( ',', '.' ) );
	    if( (hour >= 0) && (minute >= 0) && (second >= 0F) ) {
	      minute += hour * 60;
	      second += (float) (minute * 60);
	      float totalSeconds = this.recognitionData.length / frameRate;
	      idx = Math.round( second / totalSeconds
					* this.recognitionData.length );
	      if( idx >= this.recognitionData.length ) {
		idx = this.recognitionData.length - 1;
	      }
	    }
	  }
	  catch( NumberFormatException ex ) {
	    idx = -1;
	  }
	  if( idx > 0 ) {
	    this.analysisFld.setCursorIdx( idx );
	    scrollToIdx( idx );
	    break;
	  } else {
	    UIUtil.showErrorMsg(
		this,
		"Ung\u00FCltige Eingabe!\n"
			+ "Geben Sie bitte einen Zeitpunkt in einer"
			+ " der folgenden Formen ein:\n"
			+ "    Stunde:Minute:Sekunde\n"
			+ "    Minute:Sekunde\n"
			+ "    Sekunde\n\n"
			+ "Die Sekunde kann auch mit Nachkommastellen"
			+ " angegeben werden." );
	  }
	}
      }
    }
  }


  private void doPhaseDataSaveAs()
  {
    if( (this.audioFmt != null) && (this.recognitionData != null) ) {
      byte[] phaseData = new byte[ this.recognitionData.length ];
      for( int i = 0; i < phaseData.length; i++ ) {
	phaseData[ i ] = (byte) ((this.recognitionData[ i ]
					& AudioProcessor.PHASE_MASK) != 0 ?
								0xFF : 0);
      }
      File file = saveAudioFileAs(
			new AudioFormat(
				this.audioFmt.getFrameRate(),
				8,
				1,
				false,
				false ),
			phaseData,
			0,
			phaseData.length,
			"Ermittelte Phasenlagedaten speichern",
			this.recentPhaseDataDir );
      if( file != null ) {
	this.recentPhaseDataDir = file.getParentFile();
      }
    }
  }


  private void doSelectedAudioDataPlay()
  {
    if( (this.audioFmt != null) && (this.audioData != null)
	&& (this.fromToIdx >= 0) && (this.cursorIdx >= 0) )
    {
      int idx = Math.min( this.fromToIdx, this.cursorIdx );
      int len = Math.abs( this.fromToIdx - this.cursorIdx );
      if( len > 0 ) {
	len++;
	playAudio(
		this.audioFmt,
		this.audioData,
		idx * this.audioFmt.getFrameSize(),
		len * this.audioFmt.getFrameSize() );
      }
    }
  }


  private void doSelectedAudioDataSaveAs()
  {
    if( (this.audioFmt != null) && (this.audioData != null)
	&& (this.fromToIdx >= 0) && (this.cursorIdx >= 0) )
    {
      int idx = Math.min( this.fromToIdx, this.cursorIdx );
      int len = Math.abs( this.fromToIdx - this.cursorIdx );
      if( len > 0 ) {
	len++;
	File file = saveAudioFileAs(
			this.audioFmt,
			this.audioData,
			idx * this.audioFmt.getFrameSize(),
			len * this.audioFmt.getFrameSize(),
			"Ausgew\u00E4hlte Audiodaten speichern",
			this.recentAudioDataDir );
	if( file != null ) {
	  this.recentAudioDataDir = file.getParentFile();
	}
      }
    }
  }


	/* --- private Methoden --- */

  private void clear()
  {
    this.audioFmt        = null;
    this.audioData       = null;
    this.recognitionData = null;
    this.curFile         = null;
    this.analysisFld.clear();
    updTitle();
  }


  private void cursorIdxChangedInternal( int cursorIdx, int fromToIdx )
  {
    boolean hasSelection = false;
    this.cursorIdx       = -1;
    this.fromToIdx       = -1;
    this.fldCursorFrameIdx.setText( "" );
    this.fldCursorTime.setText( "" );
    this.fldFromToFrameIdx.setText( "" );
    this.fldFromToTime.setText( "" );
    if( (this.audioData != null) && (this.recognitionData != null) ) {
      if( (cursorIdx >= 0) && (cursorIdx < this.recognitionData.length) ) {
	this.cursorIdx = cursorIdx;
	this.fldCursorFrameIdx.setText( String.valueOf( cursorIdx ) );
	this.fldCursorTime.setText( getTimeText( cursorIdx ) );
	if( (fromToIdx >= 0)
	    && (fromToIdx < this.recognitionData.length) )
	{
	  this.fromToIdx = fromToIdx;
	  this.fldFromToFrameIdx.setText( String.valueOf( fromToIdx ) );
	  this.fldFromToTime.setText( getTimeText( fromToIdx ) );
	  if( fromToIdx != cursorIdx ) {
	    hasSelection = true;
	  }
	}
      }
    }
    if( this.mnuSelectedAudioDataSaveAs != null ) {
      this.mnuSelectedAudioDataSaveAs.setEnabled( hasSelection );
    }
    this.mnuSelectedAudioDataPlay.setEnabled( hasSelection );
    this.mnuMarkCurPos.setEnabled( this.cursorIdx >= 0 );
    this.mnuUnmark.setEnabled( hasSelection );
  }


  private String getTimeText( int frameIdx )
  {
    String rv = "";
    if( (this.audioFmt != null)
	&& (this.audioData != null)
	&& (this.recognitionData != null) )
    {
      if( (frameIdx >= 0) && (frameIdx < this.recognitionData.length) ) {
	float frameRate = this.audioFmt.getFrameRate();
	if( frameRate >= 1F ) {
	  int millis  = Math.round( (float) frameIdx / frameRate * 1000F );
	  int seconds = millis / 1000;
	  int minutes = seconds / 60;
	  int hours   = minutes / 60;
	  rv = String.format(
		"%1d:%02d:%02d%c%03d",
		hours,
		minutes % 60,
		seconds % 60,
		DecimalFormatSymbols.getInstance().getDecimalSeparator(),
		millis % 1000 );
	}
      }
    }
    return rv;
  }


  private void playAudio(
			AudioFormat audioFmt,
			byte[]      audioData,
			int         audioIdx,
			int         audioLen )
  {
    if( (audioIdx + audioLen) > audioData.length ) {
      audioLen = audioData.length - audioIdx;
    }
    int frameSize = audioFmt.getFrameSize();
    if( (frameSize > 0) && (audioIdx >= 0) && (audioLen > 0) ) {
      int frameIdx = audioIdx / frameSize;
      int frameLen = audioLen / frameSize;
      if( frameLen > 0 ) {
	AudioPlayFrm.open(
			this,
			audioData,
			frameIdx,
			frameLen,
			audioFmt );
      }
    }
  }


  private File saveAudioFileAs(
			AudioFormat audioFmt,
			byte[]      audioData,
			int         audioIdx,
			int         audioLen,
			String      title,
			File        recentDir )
  {
    File rv = null;
    if( (audioIdx + audioLen) > audioData.length ) {
      audioLen = audioData.length - audioIdx;
    }
    int frameSize = audioFmt.getFrameSize();
    if( (frameSize > 0) && (audioIdx >= 0) && (audioLen > 0) ) {
      int frameIdx = audioIdx / frameSize;
      int frameLen = audioLen / frameSize;
      if( frameLen > 0 ) {
	File file = UIUtil.showSaveFileDlg(
				this,
				title,
				recentDir,
				null,
				this.audioFileFilter,
				null );
	if( file != null ) {
	  try {
	    AudioFileFormat.Type fileType = UIUtil.getAudioFileType(
							file.getName() );
	    try( OutputStream out = new BufferedOutputStream(
					new FileOutputStream( file ) ) )
	    {
	      try( AudioInputStream in = new AudioInputStream(
				new ByteArrayInputStream(
						audioData,
						frameIdx * frameSize,
						frameLen * frameSize ),
				audioFmt,
				frameLen ) )
	      {
		AudioSystem.write( in, fileType, out );
	      }
	    }
	    rv = file;
	  }
	  catch( IOException ex ) {
	    UIUtil.showErrorMsg( this, ex.getMessage() );
	  }
	}
      }
    }
    return rv;
  }


  private void scrollToIdx( int idx )
  {
    if( this.recognitionData != null ) {
      if( this.recognitionData.length > 0 ) {
	int       wAnalysis = this.analysisFld.getWidth();
	Rectangle vpRect    = this.scrollPane.getViewport().getViewRect();
	if( vpRect != null ) {
	  float fPos = (float) idx / (float) this.recognitionData.length;
	  int   x    = Math.round( fPos * (float) wAnalysis )
				- (vpRect.width / 2);
	  if( x > (wAnalysis - vpRect.width) ) {
	    x = wAnalysis - vpRect.width;
	  }
	  if( x < 0 ) {
	    x = 0;
	  }
	  this.scrollPane.getViewport().setViewPosition(
						new Point( x, vpRect.y ) );
	}
      }
    }
  }


  private void updFileFieldsEnabled()
  {
    boolean state = false;
    if( (this.audioFmt != null)
	&& (this.audioData != null)
	&& (this.recognitionData != null) )
    {
      if( (this.audioData.length > 0)
	  && (this.recognitionData.length > 0) )
      {
	state = true;
      }
    }
    this.mnuAnalysisSaveAs.setEnabled( state );
    if( this.mnuAudioDataSaveAs != null ) {
      this.mnuAudioDataSaveAs.setEnabled( state );
    }
    if( this.mnuPhaseDataSaveAs != null ) {
      this.mnuPhaseDataSaveAs.setEnabled( state );
    }
    this.mnuAudioDataInfo.setEnabled( state );
    this.mnuAudioDataPlay.setEnabled( state );
    this.mnuGotoFramePos.setEnabled( state );
    this.mnuGotoTimestamp.setEnabled( state );
    this.labelCursorFrameIdx.setEnabled( state );
    this.labelCursorTime.setEnabled( state );
    this.labelFromToFrameIdx.setEnabled( state );
    this.labelFromToTime.setEnabled( state );
    this.fldCursorFrameIdx.setEnabled( state );
    this.fldCursorTime.setEnabled( state );
    this.fldFromToFrameIdx.setEnabled( state );
    this.fldFromToTime.setEnabled( state );
  }


  private void updScaleFieldsEnabled()
  {
    this.btnScalePlus.setEnabled( this.analysisFld.canSalePlus() );
    this.btnScaleMinus.setEnabled( this.analysisFld.canSaleMinus() );
  }


  private void updTitle()
  {
    StringBuilder buf = new StringBuilder( 128 );
    buf.append( Main.APPNAME );
    buf.append( " Analyse" );
    if( (this.audioFmt != null)
	&& (this.audioData != null)
	&& (this.recognitionData != null) )
    {
      if( this.curFile != null ) {
	String fName = this.curFile.getName();
	if( fName != null ) {
	  if( !fName.isEmpty() ) {
	    buf.append( ": " );
	    buf.append( fName );
	  }
	}
      }
    } else {
      buf.append( ": Keine Analysedaten vorhanden" );
    }
    setTitle( buf.toString() );
  }
}
