/*
 * (c) 2023-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Komponente zur Anzeige der Analysedaten
 */

package jkcload.ui;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Stroke;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;
import javax.sound.sampled.AudioFormat;
import javax.swing.JComponent;
import javax.swing.JScrollPane;
import javax.swing.JViewport;
import javax.swing.Scrollable;
import javax.swing.SwingConstants;
import jkcload.audio.AudioProcessor;


public class AnalysisFld extends JComponent implements
						MouseListener,
						MouseMotionListener,
						MouseWheelListener,
						Scrollable
{
  public static final Color BG_COLOR = Color.WHITE;

  private static final int AUDIO_Y        = 130;
  private static final int CURSOR_Y1      = 0;
  private static final int CURSOR_Y2      = 265;
  private static final int PHASE1_Y       = 150;
  private static final int PHASE0_Y       = 170;
  private static final int BLOCK_STATUS_Y = 240;
  private static final int EXPLANATION_Y  = 300;
  private static final int FONT_H         = 12;
  private static final int PREF_H         = EXPLANATION_Y + FONT_H + 20;
  private static final int PREF_W         = 500;
  private static final int MIN_DOTS_PER_SAMPLE = -8;

  private static final double RECOGN_Y1 = 10.0;
  private static final double RECOGN_Y2 = 180.0;


  private class RowHeaderView extends JComponent
  {
    private Font font;

    private RowHeaderView()
    {
      this.font = new Font( Font.SANS_SERIF, Font.PLAIN, FONT_H );
      setPreferredSize( new Dimension( ((FONT_H + 2) * 2) + 8, PREF_H ) );
    }

    @Override
    public void paintComponent( Graphics g )
    {
      Rectangle area = getVisibleArea();
      if( (area.width > 0) && (area.height > 0) ) {
	g = g.create();
	g.setPaintMode();
	g.setColor( Color.LIGHT_GRAY );
	g.fillRect( area.x, area.y, area.width, area.height );
	if( g instanceof Graphics2D ) {
	  Graphics2D g2d = (Graphics2D) g;
	  g2d.rotate( -Math.PI / 2.0 );
	  g2d.setColor( Color.BLACK );
	  g2d.setFont( this.font );
	  final int y1 = FONT_H + 2;
	  final int y2 = 2 * y1;
	  int x = 64 - AUDIO_Y;
	  drawCenteredString( g, "Gelesene", x, y1 );
	  drawCenteredString( g, "Audiodaten", x, y2 );
	  x = -PHASE1_Y - ((PHASE0_Y - PHASE1_Y) / 2);
	  drawCenteredString( g, "Erkannte", x, y1 );
	  drawCenteredString( g, "Phasenlage", x, y2 );
	  x = -BLOCK_STATUS_Y;
	  drawCenteredString( g, "Gelesene", x, y1 );
	  drawCenteredString( g, "Bl\u00F6cke", x, y2 );
	}
	g.dispose();
      }
    }
  };


  private AnalysisFrm analysisFrm;
  private byte[]      audioData;
  private byte[]      recognitionData;
  private float       frameRate;
  private float       midValue;
  private int         maxSamplesPerDot;
  private int         samplesPerDot;
  private int         cursorIdx;
  private int         playIdx;
  private int         fromToIdx;
  private boolean     fromToFixed;
  private boolean     hasContent;
  private boolean     notified;
  private Color       colorAmplitudeMain;
  private Color       colorAmplitudeTop;
  private Color       colorBlockErr;
  private Color       colorBlockOK;
  private Color       colorMark;
  private Color       colorWavePilot;
  private Color       colorWaveDelim;
  private Color       colorWaveBit0;
  private Color       colorWaveBit1;
  private Font        font;
  private Stroke      strokeBar;
  private Stroke      strokeCursor;
  private Stroke      strokePlay;
  private Stroke      strokeThin;
  private Stroke      strokeThick;


  public AnalysisFld( AnalysisFrm analysisFrm )
  {
    this.analysisFrm        = analysisFrm;
    this.notified           = false;
    this.cursorIdx          = -1;
    this.playIdx            = -1;
    this.colorAmplitudeMain = new Color( 0, 0, 0xFF );
    this.colorAmplitudeTop  = new Color( 0, 0, 0xBF );
    this.colorBlockErr      = Color.RED;
    this.colorBlockOK       = Color.GREEN;
    this.colorMark          = new Color( 0xFFBFBFFF );
    this.colorWavePilot     = Color.YELLOW;
    this.colorWaveDelim     = Color.CYAN;
    this.colorWaveBit0      = Color.ORANGE;
    this.colorWaveBit1      = Color.PINK;
    this.font      = new Font( Font.SANS_SERIF, Font.PLAIN, FONT_H );
    this.strokeBar = new BasicStroke(
				20F,
				BasicStroke.CAP_BUTT,
				BasicStroke.JOIN_BEVEL );
    this.strokeCursor = new BasicStroke(
				0.3F,
				BasicStroke.CAP_BUTT,
				BasicStroke.JOIN_BEVEL );
    this.strokePlay = new BasicStroke(
				0.5F,
				BasicStroke.CAP_BUTT,
				BasicStroke.JOIN_BEVEL );
    this.strokeThin = new BasicStroke(
				0.1F,
				BasicStroke.CAP_ROUND,
				BasicStroke.JOIN_ROUND );
    this.strokeThick = new BasicStroke(
				1.0F,
				BasicStroke.CAP_ROUND,
				BasicStroke.JOIN_ROUND );
    setCursor( Cursor.getPredefinedCursor( Cursor.TEXT_CURSOR ) );
    setPreferredSize( new Dimension( PREF_W, PREF_H ) );
    clear();
  }


  public boolean canSaleMinus()
  {
    return this.hasContent && (this.samplesPerDot < this.maxSamplesPerDot);
  }


  public boolean canSalePlus()
  {
    return this.hasContent && (this.samplesPerDot > MIN_DOTS_PER_SAMPLE);
  }


  public JComponent createRowHeaderView()
  {
    return new RowHeaderView();
  }


  public void clear()
  {
    this.audioData        = null;
    this.recognitionData  = null;
    this.frameRate        = 0F;
    this.maxSamplesPerDot = 0;
    this.samplesPerDot    = 0;
    this.midValue         = -1F;
    this.fromToIdx        = -1;
    this.fromToFixed      = false;
    this.hasContent       = false;
    revalidate();
    repaint();
    setCursorIdx( -1 );
  }


  public void scaleMinus()
  {
    scale( -1 );
  }


  public void scalePlus()
  {
    scale( 1 );
  }


  public void setAnalysisData(
			AudioFormat audioFmt,
			byte[]      audioData,
			byte[]      recognitionData )
  {
    if( (audioFmt != null)
	&& (audioData != null)
	&& (recognitionData != null) )
    {
      float frameRate         = audioFmt.getFrameRate();
      int   sampleSizeInBits  = audioFmt.getSampleSizeInBits();
      int   sampleSizeInBytes = (sampleSizeInBits + 7) / 8;
      if( (frameRate > 0F)
	  && (sampleSizeInBits > 0)
	  && (sampleSizeInBytes > 0)
	  && (audioData.length > 0)
	  && (recognitionData.length > 0) )
      {
	int frameCount = audioData.length / sampleSizeInBytes;
	if( frameCount > 0 ) {
	  boolean bigEndian  = audioFmt.isBigEndian();
	  boolean dataSigned = audioFmt.getEncoding().equals(
					AudioFormat.Encoding.PCM_SIGNED );

	  // Audiodaten auf 7 Bit (1 Byte unsigned) umrechnen
	  this.audioData      = new byte[ frameCount ];
	  int  shiftWidth     = sampleSizeInBits - 7;	// 7-Bit
	  int  sampleBitMask  = ((1 << sampleSizeInBits) - 1);
	  int  sampleSignMask = (1 << (sampleSizeInBits - 1));
	  int  nValues        = 0;
	  long totalValue     = 0;
	  int srcIdx          = 0;
	  int dstIdx          = 0;
	  while( (srcIdx < audioData.length)
		 && (dstIdx < this.audioData.length) )
	  {
	    int v = 0;
	    if( bigEndian ) {
	      for( int i = 0; i < sampleSizeInBytes; i++ ) {
		v = (v << 8) | ((int) audioData[ srcIdx + i ] & 0xFF);
	      }
	    } else {
	      for( int i = sampleSizeInBytes - 1; i >= 0; --i ) {
		v = (v << 8) | ((int) audioData[ srcIdx + i ] & 0xFF);
	      }
	    }
	    v &= sampleBitMask;
	    if( dataSigned ) {
	      v ^= sampleSignMask;
	    }
	    if( shiftWidth < 0 ) {
	      v <<= shiftWidth;
	    } else if( shiftWidth > 0 ) {
	      v >>= shiftWidth;
	    }
	    this.audioData[ dstIdx++ ] = (byte) v;
	    totalValue += v;
	    nValues++;
	    srcIdx += sampleSizeInBytes;
	  }
	  this.recognitionData = recognitionData;
	  this.frameRate       = frameRate;
	  if( nValues > 0 ) {
	    this.midValue = (float) Math.round(
				(double) totalValue / (double) nValues );
	  }

	  int w = getVisibleArea().width;
	  if( w < 1 ) {
	    w = 1;
	  }
	  int samplesPerDot     = recognitionData.length / w;
	  this.maxSamplesPerDot = 1;
	  while( this.maxSamplesPerDot < samplesPerDot ) {
	    this.maxSamplesPerDot <<= 1;
	  }
	  this.samplesPerDot = this.maxSamplesPerDot / 4;
	  this.hasContent    = true;
	  setCursorIdx( -1 );
	  updPrefSize();
	  fireMoveViewportToFrameIdxAt( 0, 0 );
	}
      }
    }
  }


  public void setCursorIdx( int idx )
  {
    if( idx != this.cursorIdx ) {
      this.cursorIdx = idx;
      this.analysisFrm.cursorIdxChanged( idx, this.fromToIdx );
      repaint();
    }
  }


  public void setFromTo( int idx )
  {
    if( (idx >= 0) && (this.recognitionData != null) ) {
      if( idx > this.recognitionData.length ) {
	idx = this.recognitionData.length;
      }
    }
    this.fromToIdx   = idx;
    this.fromToFixed = (idx >= 0);
    this.analysisFrm.cursorIdxChanged( this.cursorIdx, this.fromToIdx );
    repaint();
  }


  public void setPlayPosition( int idx )
  {
    if( idx != this.playIdx ) {
      this.playIdx = idx;
      repaint();
    }
  }


  public void unmark()
  {
    this.fromToIdx   = -1;
    this.fromToFixed = false;
    this.analysisFrm.cursorIdxChanged( this.cursorIdx, this.fromToIdx );
    repaint();
  }


	/* --- MouseListener --- */

  @Override
  public void mouseClicked( MouseEvent e )
  {
    // leer
  }


  @Override
  public void mouseEntered( MouseEvent e )
  {
    // leer
  }


  @Override
  public void mouseExited( MouseEvent e )
  {
    // leer
  }


  @Override
  public void mousePressed( MouseEvent e )
  {
    if( e.getButton() == MouseEvent.BUTTON1 ) {
      if( !this.fromToFixed ) {
	this.fromToIdx = -1;
      }
      setCursorX( e.getX() );
      e.consume();
    }
  }


  @Override
  public void mouseReleased( MouseEvent e )
  {
    // leer
  }


	/* --- MouseMotionListener --- */

  @Override
  public void mouseDragged( MouseEvent e )
  {
    if( this.fromToIdx < 0 ) {
      this.fromToIdx = this.cursorIdx;
    }
    this.fromToFixed = false;
    setCursorX( e.getX() );
    e.consume();
  }


  @Override
  public void mouseMoved( MouseEvent e )
  {
    // leer
  }


	/* --- MouseWheelListener --- */

  @Override
  public void mouseWheelMoved( MouseWheelEvent e )
  {
    if( e.getScrollType() == MouseWheelEvent.WHEEL_UNIT_SCROLL ) {
      scale( e.getWheelRotation() );
    }
    e.consume();
  }


	/* --- Scrollable --- */

  @Override
  public Dimension getPreferredScrollableViewportSize()
  {
    return getPreferredSize();
  }


  @Override
  public int getScrollableBlockIncrement(
			Rectangle visibleRect,
			int       orientation,
			int       direction )
  {
    int rv = 1;
    switch( orientation ) {
      case SwingConstants.HORIZONTAL:
	rv = visibleRect.width / 4;
	break;
      case SwingConstants.VERTICAL:
	rv = Math.min( visibleRect.height, PREF_H / 10 );
	break;
    }
    return rv > 0 ? rv : 1;
  }


  @Override
  public boolean getScrollableTracksViewportHeight()
  {
    return false;
  }


  @Override
  public boolean getScrollableTracksViewportWidth()
  {
    return false;
  }


  @Override
  public int getScrollableUnitIncrement(
			Rectangle visibleRect,
			int       orientation,
			int       direction )
  {
    return getScrollableBlockIncrement(
				visibleRect,
				orientation,
				direction );
  }


	/* --- ueberschriebene Methoden --- */

  @Override
  public void addNotify()
  {
    super.addNotify();
    if( !this.notified ) {
      this.notified = true;
      addMouseListener( this );
      addMouseMotionListener( this );
      addMouseWheelListener( this );
    }
  }


  @Override
  public void removeNotify()
  {
    super.removeNotify();
    if( this.notified ) {
      this.notified = false;
      removeMouseListener( this );
      removeMouseMotionListener( this );
      removeMouseWheelListener( this );
    }
  }


  @Override
  public void paintComponent( Graphics g )
  {
    g = g.create();
    Rectangle area = getVisibleArea();
    if( (area.width > 0) && (area.height > 0) ) {
      g.setPaintMode();
      g.setColor( Color.WHITE );
      g.fillRect( area.x, area.y, area.width, area.height );
      if( (g instanceof Graphics2D)
	  && (this.frameRate > 0F)
	  && (this.audioData != null)
	  && (this.recognitionData != null) )
      {
	Graphics2D g2d = (Graphics2D) g;

	// Analyse zeichnen
	if( this.samplesPerDot > 50 ) {
	  drawAnalysisCompressed( g2d, area );
	} else {
	  drawAnalysisDetailed( g2d, area );
	}

	// Auswahl zeichnen
	int xCursor = getCursorX();
	if( (this.cursorIdx >= 0) && (this.fromToIdx >= 0)
	    && (this.cursorIdx != this.fromToIdx) )
	{
	  int x1 = getXByFrameIdx( this.fromToIdx );
	  int x2 = xCursor;
	  if( x2 < x1 ) {
	    x2 = x1;
	    x1 = xCursor;
	  }
	  if( x1 < area.x ) {
	    x1 = area.x;
	  }
	  if( x2 > (area.x + area.width) ) {
	    x2 = area.x + area.width;
	  }
	  g2d.setXORMode( Color.WHITE );
	  g2d.setColor( this.colorMark );
	  g2d.fillRect(
		x1,
		CURSOR_Y1,
		x2 - x1,
		CURSOR_Y2 - CURSOR_Y1 );
	}

	// Cursor-Linie zeichnen
	if( (xCursor >= area.x) && (xCursor < (area.x + area.width)) ) {
	  g2d.setPaintMode();
	  g2d.setColor( Color.BLACK );
	  g2d.setStroke( this.strokeCursor );
	  g2d.drawLine( xCursor, CURSOR_Y1, xCursor, CURSOR_Y2 );
	}

	// Abspielposition zeichnen
	int xPlay = getXByFrameIdx( this.playIdx );
	if( (xPlay >= area.x) && (xPlay < (area.x + area.width)) ) {
	  g2d.setPaintMode();
	  g2d.setColor( Color.MAGENTA );
	  g2d.setStroke( this.strokePlay );
	  g2d.drawLine( xPlay, CURSOR_Y1, xPlay, CURSOR_Y2 );
	}
      }
    }
    g.dispose();
  }


	/* --- private Methoden --- */

  private void drawAnalysisCompressed( Graphics2D g2d, Rectangle area )
  {
    int idx = area.x * this.samplesPerDot;
    if( idx < 0 ) {
      idx = 0;
    }
    if( idx < this.audioData.length ) {
      drawAudioMidLine( g2d, area );
      int    n             = 0;
      int    v             = 0;
      int    vMin          = 0;
      int    vMax          = 0;
      double vDiff         = 0.0;
      double vTotalDiffMin = 0;
      double vTotalDiffMax = 0;
      double x             = area.getX();
      double xLast         = x;
      double xStep         = 0.1;
      double xEnd          = x + area.getWidth();
      double yAudio        = (double) AUDIO_Y;
      double yMid          = yAudio - this.midValue;
      int    idx2          = 0;
      while( (idx < this.audioData.length) && (x < xEnd) ) {
	vTotalDiffMin = 0.0;
	vTotalDiffMax = 0.0;
	v    = this.audioData[ idx ] & 0xFF;
	n    = 0;
	vMin = v;
	vMax = v;
	idx2 = (int) Math.round( (x + xStep) * this.samplesPerDot );
	if( idx2 > this.audioData.length ) {
	  idx2 = this.audioData.length;
	}
	while( (idx < idx2) && (x < xEnd) ) {
	  v = this.audioData[ idx++ ] & 0xFF;
	  if( v < vMin ) {
	    vMin = v;
	  }
	  if( v > vMax ) {
	    vMax = v;
	  }
	  vDiff = (double) v - this.midValue;
	  if( vDiff < 0 ) {
	    vTotalDiffMin += vDiff;
	  } else {
	    vTotalDiffMax += vDiff;
	  }
	  n++;
	}
	if( n > 0 ) {
	  g2d.setColor( this.colorAmplitudeTop );
	  g2d.setStroke( this.strokeThin );
	  g2d.draw( new Line2D.Double(
				x,
				yAudio - (double) vMin,
				x,
				yAudio - (double) vMax ) );
	  g2d.setColor( this.colorAmplitudeMain );
	  g2d.setStroke( this.strokeThick );
	  g2d.draw( new Line2D.Double(
				xLast,
				yMid - (vTotalDiffMin / (double) n),
				x,
				yMid - (vTotalDiffMax / (double) n) ) );
	}
	xLast = x;
	x += xStep;
      }
      if( this.samplesPerDot > 0 ) {
	xStep = 1.0 / (double) this.samplesPerDot;
	if( Double.isFinite( xStep ) ) {
	  idx = area.x * this.samplesPerDot;
	  drawBlockStatus( g2d, area, xStep, idx > 0 ? idx : 0 );
	}
      }
      drawBlockStatusExplanation( g2d, area );
    }
  }


  private void drawAnalysisDetailed( Graphics2D g2d, Rectangle area )
  {
    if( this.samplesPerDot > 0 ) {
      double xStep = 1.0 / (double) this.samplesPerDot;
      int    idx   = area.x * this.samplesPerDot;
      if( idx < 0 ) {
	idx = 0;
      }
      drawRecognition( g2d, area, xStep, idx );
      drawAudio( g2d, area, xStep, idx );
      drawPhases( g2d, area, xStep, idx );
      drawBlockStatus( g2d, area, xStep, idx );
    } else {
      int dps = 2 - this.samplesPerDot;
      int idx = area.x / dps;
      if( idx < 0 ) {
	idx = 0;
      }
      double xStep = (double) dps;
      drawRecognition( g2d, area, xStep, idx );
      drawAudio( g2d, area, xStep, idx );
      drawPhases( g2d, area, xStep, idx );
      drawBlockStatus( g2d, area, xStep, idx );
    }
    drawExplanation(
		g2d,
		drawBlockStatusExplanation( g2d, area ),
		new Color[] {
			this.colorWavePilot,
			this.colorWaveDelim,
			this.colorWaveBit0,
			this.colorWaveBit1 },
		new String[] {
			"Vorton",
			"Trennung",
			"0-Bit",
			"1-Bit" } );
  }


  private void drawAudio(
			Graphics2D g2d,
			Rectangle  area,
			double     xStep,
			int        idx )
  {
    drawAudioMidLine( g2d, area );
    g2d.setColor( this.colorAmplitudeMain );
    double x    = area.getX();
    double xEnd = x + area.getWidth();
    if( idx < this.audioData.length ) {
      double xLast = x;
      x += xStep;
      int v     = this.audioData[ idx++ ] & 0xFF;
      int yLast = AUDIO_Y - v;
      int y     = yLast;
      while( (idx < this.audioData.length) && (x < xEnd) ) {
	v = this.audioData[ idx++ ] & 0xFF;
	y = AUDIO_Y - v;
	if( this.samplesPerDot > 1 ) {
	  g2d.draw( new Line2D.Double(
				xLast,
				(double) yLast,
				x,
				(double) y ) );
	} else {
	  g2d.draw( new Line2D.Double(
				xLast,
				(double) yLast,
				x,
				(double) yLast ) );
	  g2d.draw( new Line2D.Double(
				x,
				(double) yLast,
				x,
				(double) y ) );
	}
	xLast = x;
	yLast = y;
	x += xStep;
      }
    }
  }


  private void drawAudioMidLine( Graphics2D g2d, Rectangle  area )
  {
    if( this.midValue > 0F ) {
      g2d.setStroke( this.strokeThin );
      g2d.setColor( Color.BLACK );
      g2d.draw( new Line2D.Float(
			(float) area.x,
			(float) AUDIO_Y - this.midValue,
			(float) area.x + area.width,
			(float) AUDIO_Y - this.midValue ) );
    }
  }


  private void drawBlockStatus(
			Graphics2D g2d,
			Rectangle  area,
			double     xStep,
			int        idx )
  {
    g2d.setStroke( this.strokeBar );
    double x = area.getX();
    if( idx < this.recognitionData.length ) {
      final double y = (double) BLOCK_STATUS_Y;
      double xEnd    = x + area.getWidth();
      double xLast   = x;
      x += xStep;
      int     v             = this.recognitionData[ idx++ ];
      boolean blockRead     = ((v & AudioProcessor.BLOCK_READ) != 0);
      boolean blockErr      = ((v & AudioProcessor.BLOCK_ERROR) != 0);
      boolean lastBlockRead = blockRead;
      boolean lastBlockErr  = blockErr;
      while( (idx < this.recognitionData.length) && (x < xEnd) ) {
	v         = this.recognitionData[ idx++ ];
	blockRead = ((v & AudioProcessor.BLOCK_READ) != 0);
	blockErr  = ((v & AudioProcessor.BLOCK_ERROR) != 0);
	if( (blockRead != lastBlockRead) || (blockErr != lastBlockErr) ) {
	  if( lastBlockRead ) {
	    g2d.setColor( lastBlockErr ?
				this.colorBlockErr
				: this.colorBlockOK );
	    g2d.draw( new Line2D.Double( xLast, y, x, y ) );
	  }
	  lastBlockRead = blockRead;
	  lastBlockErr  = blockErr;
	  xLast = x;
	}
	x += xStep;
      }
      if( x > xLast ) {
	if( lastBlockRead ) {
	  g2d.setColor( lastBlockErr ?
				this.colorBlockErr
				: this.colorBlockOK );
	  g2d.draw( new Line2D.Double( xLast, y, x, y ) );
	}
      }
    }
  }


  private int drawBlockStatusExplanation( Graphics2D g2d, Rectangle area )
  {
    return drawExplanation(
			g2d,
			area.x + 20,
			new Color[] {
				this.colorBlockOK,
				this.colorBlockErr },
			new String[] {
				"Block fehlerfrei gelesen", 
				"Block mit Fehler gelesen" } );
  }


  private int drawExplanation(
			Graphics2D g2d,
			int        x,
			Color[]    colors,
			String[]   explanations )
  {
    g2d.setFont( this.font );
    FontMetrics fm = g2d.getFontMetrics();
    if( fm != null ) {
      float yBar  = (float) EXPLANATION_Y;
      int   yText = EXPLANATION_Y + (FONT_H / 2);
      g2d.setStroke( this.strokeBar );
      for( int i = 0;
	   (i < colors.length) && (i < explanations.length);
	   i++ )
      {
	g2d.setColor( colors[ i ] );
	g2d.draw( new Line2D.Float( x, yBar, x + 20, yBar ) );
	x += 25;
	g2d.setColor( Color.BLACK );
	g2d.drawString( explanations[ i ], x, yText );
	x += fm.stringWidth( explanations[ i ] );
	x += 20;
      }
    }
    return x;
  }


  private void drawPhases(
			Graphics2D g2d,
			Rectangle  area,
			double     xStep,
			int        idx )
  {
    g2d.setColor( this.colorAmplitudeMain );
    double x = area.getX();
    if( idx < this.recognitionData.length ) {
      double xEnd  = x + area.getWidth();
      double xLast = x;
      x += xStep;
      boolean pLast = ((this.recognitionData[ idx++ ]
					& AudioProcessor.PHASE_MASK) != 0);
      boolean phase = pLast;
      double  yLast = (phase ? (double) PHASE1_Y : (double) PHASE0_Y);
      double  y     = yLast;
      while( (idx < this.recognitionData.length) && (x < xEnd) ) {
	phase = ((this.recognitionData[ idx++ ]
				& AudioProcessor.PHASE_MASK) != 0);
	if( phase != pLast ) {
	  g2d.draw( new Line2D.Double( xLast, yLast, x, yLast ) );
	  y = (phase ? (double) PHASE1_Y : (double) PHASE0_Y);
	  g2d.draw( new Line2D.Double(
				x,
				(double) yLast,
				x,
				(double) y ) );
	  pLast = phase;
	  xLast = x;
	  yLast = y;
	}
	x += xStep;
      }
      if( x > xLast ) {
	g2d.draw( new Line2D.Double( xLast, yLast, x, yLast ) );
      }
    }
  }


  private void drawRecognition(
			Graphics2D g2d,
			Rectangle  area,
			double     xStep,
			int        idx )
  {
    double x = area.getX();
    if( idx < this.recognitionData.length ) {
      double xEnd  = x + area.getWidth();
      double xLast = x;
      x += xStep;
      int vLast = this.recognitionData[ idx++ ] & AudioProcessor.WAVE_MASK;
      int v     = vLast;
      while( (idx < this.recognitionData.length) && (x < xEnd) ) {
	while( (idx < this.recognitionData.length) && (x < xEnd) ) {
	  x += xStep;
	  v = this.recognitionData[ idx++ ] & AudioProcessor.WAVE_MASK;
	  if( v != vLast ) {
	    break;
	  }
	}
	if( x > xLast ) {
	  Color color = null;
	  switch( vLast ) {
	    case AudioProcessor.WAVE_PILOT:
	      color = this.colorWavePilot;
	      break;
	    case AudioProcessor.WAVE_DELIM:
	      color = this.colorWaveDelim;
	      break;
	    case AudioProcessor.WAVE_BIT_0:
	      color = this.colorWaveBit0;
	      break;
	    case AudioProcessor.WAVE_BIT_1:
	      color = this.colorWaveBit1;
	      break;
	  }
	  if( color != null ) {
	    g2d.setColor( color );
	    g2d.fill( new Rectangle2D.Double(
					xLast,
					RECOGN_Y1,
					x - xLast,
					RECOGN_Y2 ) );
	  }
	  xLast = x;
	  vLast = v;
	}
      }
    }
  }


  private static void drawCenteredString(
				Graphics g,
				String   s,
				int      x,
				int      y )
  {
    if( s != null ) {
      FontMetrics fm = g.getFontMetrics();
      if( fm != null ) {
	int w = fm.stringWidth( s );
	g.drawString( s, x - (w / 2), y );
      }
    }
  }


  // View-Position anpassen: xViewFix soll auf frameIdx stehen
  private void fireMoveViewportToFrameIdxAt( int frameIdx, int xViewFix )
  {
    EventQueue.invokeLater(
		()->moveViewportToFrameIdxAt( frameIdx, xViewFix ) );
  }


  private int getCursorX()
  {
    return getXByFrameIdx( this.cursorIdx );
  }


  private int getFrameIdxByX( int x )
  {
    return this.samplesPerDot > 0 ?
			(x * this.samplesPerDot)
			: (x / (2 - this.samplesPerDot));
  }


  private Rectangle getVisibleArea()
  {
    Rectangle r = null;
    int       w = getWidth();
    int       h = getHeight();
    if( (w > 0) && (h > 0) ) {
      Component p = getParent();
      if( p != null ) {
	if( p instanceof JViewport ) {
	  r = ((JViewport) p).getViewRect();
	}
      }
    }
    return r != null ? r : new Rectangle( w, h );
  }


  private int getXByFrameIdx( int frameIdx )
  {
    int x = -1;
    if( frameIdx >= 0 ) {
      if( this.samplesPerDot > 0 ) {
	x = frameIdx / this.samplesPerDot;
      } else {
	x = frameIdx * (2 - this.samplesPerDot);
      }
    }
    return x;
  }


  private void moveViewportToFrameIdxAt( int frameIdx, int xViewFix )
  {
    Component p = getParent();
    int       x = getXByFrameIdx( frameIdx );
    if( (p != null) && (x >= 0) ) {
      if( p instanceof JViewport ) {
	int vpX = x - xViewFix;
	if( vpX < 0 ) {
	  vpX = 0;
	}
	Rectangle r = ((JViewport) p).getViewRect();
	((JViewport) p).setViewPosition( new Point( vpX, r.y ) );
      }
    }
  }


  public void scale( int amount )
  {
    if( this.hasContent ) {

      /*
       * X-Koordinate ermitteln, die fest stehen soll
       *
       * Ist der Cursor sichtbar, soll dieser fest stehen bleiben,
       * anderenfalls die Mitte des sichtbaren Bereichs
       */
      Rectangle area     = getVisibleArea();
      int       xViewFix = 0;
      int       xFix     = 0;
      int       xCursor  = getCursorX();
      if( (xCursor >= area.x) && (xCursor < (area.x + area.width)) ) {
	xFix     = xCursor;
	xViewFix = xFix - area.x;
      } else {
	xViewFix = area.width / 2;
	xFix     = area.x + xViewFix;
      }
      int frameIdx = getFrameIdxByX( xFix );

      // skalieren
      boolean changed = false;
      while( (amount > 0) && (this.samplesPerDot > MIN_DOTS_PER_SAMPLE) ) {
	if( this.samplesPerDot > 1 ) {
	  this.samplesPerDot /= 2;
	} else {
	  --this.samplesPerDot;
	}
	changed = true;
	--amount;
      }
      if( !changed ) {
	while( (amount < 0)
	       && (this.samplesPerDot < this.maxSamplesPerDot) )
	{
	  if( this.samplesPerDot < 1 ) {
	    this.samplesPerDot++;
	  } else {
	    this.samplesPerDot *= 2;
	  }
	  changed = true;
	  amount++;
	}
      }
      if( changed ) {
	updPrefSize();
	fireMoveViewportToFrameIdxAt( frameIdx, xViewFix );
      }
    }
  }


  private void setCursorX( int x )
  {
    setCursorIdx( getFrameIdxByX( x ) );
  }


  private void updPrefSize()
  {
    if( this.recognitionData != null ) {
      int w = 0;
      if( this.samplesPerDot > 0 ) {
	w = this.recognitionData.length / this.samplesPerDot;
      } else {
	w = this.recognitionData.length * (2 - this.samplesPerDot);
      }
      if( w < 1 ) {
	w = 1;
      }
      setPreferredSize( new Dimension( w, PREF_H ) );
      revalidate();
      repaint();
    }
  }
}
