/*
 * (c) 2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Verarbeitung der Audiodaten im ZX-Spectrum-Format
 */

package jkcload.audio;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


public class ZXAudioProcessor extends AudioProcessor
{
  /*
   * Pulsewerte (Halbwelle);
   *   Vorton: 610 Mikrosekunden (2168 3500000 T-States)
   *   Trennschwingung: 190 und 210 Mikrosekunden (667 und 735 T-States)
   *   0-Bit: 2 x 244 Mikrosekunden (2 x 855 T-States)
   *   1-Bit: 2 x 498 Mikrosekunden (2 x 1710 T-States)
   */
  private static final int PILOT_MICROS = 610;
  private static final int DELIM_MICROS = 200;

  // 2 x (244 + (498 - 244) / 2)
  private static final int BIT_DISTINCT_MICROS = 742;

  private ByteArrayOutputStream blockBytes;
  private ByteArrayOutputStream tapBytes;
  private ByteArrayOutputStream tzxBytes;
  private StringBuilder         activityBuf;
  private StringBuilder         nameBuf;
  private StringBuilder         logBuf;
  private byte[]                headerBytes;
  private boolean               blockCSErr;
  private boolean               fileErr;
  private int                   delimMinMicros;
  private int                   delimMaxMicros;
  private int                   pilotMinMicros;
  private int                   pilotMaxMicros;


  public ZXAudioProcessor( AudioThread audioThread, float tolerance )
  {
    super( audioThread );
    this.blockBytes  = new ByteArrayOutputStream( 0x4000 );
    this.tapBytes    = new ByteArrayOutputStream( 0x4000 );
    this.tzxBytes    = new ByteArrayOutputStream( 0x4000 );
    this.activityBuf = new StringBuilder( 32 );
    this.nameBuf     = new StringBuilder( 10 );
    this.logBuf      = new StringBuilder( 1024 );
    this.headerBytes = new byte[ 18 ];		// Blocktyp + Inhalt
    this.blockCSErr  = false;
    this.fileErr     = false;

    // Zeiten berechnen
    int absTolerance    = Math.round( (float) PILOT_MICROS * tolerance );
    this.pilotMinMicros = PILOT_MICROS - absTolerance;
    this.pilotMaxMicros = PILOT_MICROS + absTolerance;
    absTolerance        = Math.round( (float) DELIM_MICROS * tolerance );
    this.delimMinMicros = DELIM_MICROS - absTolerance;
    this.delimMaxMicros = DELIM_MICROS + absTolerance;
 }


	/* --- ueberschrieben Methoden --- */

  @Override
  public void run()
  {
    try {
      this.blockBytes.reset();
      this.tapBytes.reset();
      this.tzxBytes.reset();
      this.activityBuf.setLength( 0 );
      this.logBuf.setLength( 0 );
      this.nameBuf.setLength( 0 );
      this.blockCSErr = false;
      this.fileErr    = false;

      if( readBlock( true, 17 ) ) {
	String             format        = "ZX";
	String             fileName      = null;
	String             infoText      = null;
	int                begAddr       = -1;
	int                dataLen       = 0;
	List<String>       fileExtList   = new ArrayList<>();
	Map<String,byte[]> fileExt2Bytes = new HashMap<>();
	if( this.headerBytes[ 0 ] == 0 ) {
	  dataLen = getWord( this.headerBytes, 12 );
	  switch( this.headerBytes[ 1 ] ) {
	    case 0:
	      format = "ZX Programm";
	      break;
	    case 1:
	      format = "ZX NumArray";
	      break;
	    case 2:
	      format = "ZX CharArray";
	      break;
	    case 3:
	      format  = "ZX Code";
	      begAddr = getWord( this.headerBytes, 14 );
	      if( (begAddr == 0x4000) && (dataLen == 6912) ) {
		infoText = "Screenshot";
	      }
	      break;
	  }
	  int idx = 2;
	  while( idx < 11 ) {
	    int b = this.headerBytes[ idx++ ] & 0xFF;
	    if( (b < 0x20) || (b > 0x7E) ) {
	      break;
	    }
	    this.nameBuf.append( (char) b );
	  }
	  fileName = this.nameBuf.toString();

	  this.activityBuf.append( format );
	  this.activityBuf.append( ": " );
	  this.activityBuf.append( fileName );
	  if( this.blockCSErr ) {
	    this.activityBuf.append( ", Kopfblock fehlerhaft gelesen" );
	  }
	  writeBlock();
	  this.observer.updReadActivity( this.activityBuf.toString() );
	  if( dataLen > 0 ) {
	    this.blockBytes.reset();
	    if( readBlock( false, dataLen ) ) {
	      writeBlock();
	    } else {
	      this.logBuf.append(
		"Datenblock nicht oder nicht vollst\u00E4ndig gelesen\n" );
	    }
	  }
	  fileExtList.add( "TAP" );
	  fileExtList.add( "TZX" );
	  fileExt2Bytes.put( "TAP", this.tapBytes.toByteArray() );
	  fileExt2Bytes.put( "TZX", this.tzxBytes.toByteArray() );
	} else {
	  // kein Kopfblock -> RAW-Datei
	  format  = "ZX Rohdaten";
	  dataLen = this.blockBytes.size();
	  if( dataLen > 0 ) {
	    byte[] dataBytes = this.blockBytes.toByteArray();
	    if( dataBytes != null ) {
	      /*
	       * Aufgrund der unbekannten Laenge des Blocks
	       * wurde ohne Fehlererkennung gelesen.
	       * Dies wird nun nachgeholt.
	       */
	      if( dataBytes.length > 1 ) {
		byte checkSum = dataBytes[ 0 ];
		int  idx      = 1;
		while( idx < (dataBytes.length - 1) ) {
		  checkSum ^= dataBytes[ idx++ ];
		}
		if( dataBytes[ dataBytes.length - 1 ] != checkSum ) {
		  this.blockCSErr = true;
		  this.fileErr    = true;
		  this.logBuf.append( "\nAufgrund des fehlenden Kopfblocks\n"
			+ "musste die L\u00E4nge des Datenblocks"
			+ " selbst erkannt werden,\n"
			+ "was eine gewisse Unsicherheit bedeutet.\n"
			+ "Das letzte gelesene Byte wurde als Pr\u00FCfbyte"
			+ " gewertet,\n"
			+ "doch dieses stimmt nicht mit dem berechneten Wert"
			+ " \u00FCberein.\n" );
		}
	      }
	    }
	    fileExtList.add( "RAW" );
	    fileExt2Bytes.put( "RAW", dataBytes );
	  }
	}
	if( !fileExtList.isEmpty() ) {
	  this.observer.fileRead(
			format,
			begAddr,
			dataLen,
			-1,
			fileName,
			infoText,
			fileExtList,
			fileExt2Bytes,
			this.logBuf.toString(),
			this.fileErr );
	}
      }
    }
    catch( Exception ex ) {
      this.observer.errorOccured( null, ex );
    }
  }


	/* --- private Methoden --- */

  private boolean matchesDelimMicros( int micros )
  {
    return (micros >= this.delimMinMicros)
	   && (micros <= this.delimMaxMicros);
  }


  private boolean matchesPilotMicros( int micros )
  {
    return (micros >= this.pilotMinMicros)
	   && (micros <= this.pilotMaxMicros);
  }


  private boolean readBlock( boolean headerBlock, int blockLen )
  {
    this.blockBytes.reset();
    this.blockCSErr = false;

    boolean status = false;
    while( this.audioThread.isIOEnabled() ) {
      this.audioThread.fillUpRecognizedWave( NONE );

      // Vorton finden
      int micros = this.audioThread.readMicrosTillPhaseChange();
      if( !matchesPilotMicros( micros ) ) {
	continue;
      }

      // Vorton uebergehen
      while( this.audioThread.isIOEnabled() ) {
	this.audioThread.markBlockBegin();
	this.audioThread.fillUpRecognizedWave( WAVE_PILOT );
	micros = this.audioThread.readMicrosTillPhaseChange();
	if( !matchesPilotMicros( micros ) ) {
	  break;
	}
      }

      // Trennschwingung
      if( !matchesDelimMicros( micros ) ) {
	continue;
      }
      this.audioThread.fillUpRecognizedWave( WAVE_DELIM );
      micros = this.audioThread.readMicrosTillPhaseChange();
      if( !matchesDelimMicros( micros ) ) {
	continue;
      }
      this.audioThread.fillUpRecognizedWave( WAVE_DELIM );

      // Blocktyp lesen
      if( !this.audioThread.isIOEnabled() ) {
	break;
      }
      int b = readByte( false );
      if( headerBlock ) {
	this.headerBytes[ 0 ] = (byte) b;
	if( b != 0 ) {
	  /*
	   * kein Kopfblock -> Block bis zum Ende lesen
	   * Dabei mindestens die Groesse eines Kopfblocks
	   * als gueltige Blockgroesse ansehen
	   */
	  while( b >= 0 ) {
	    this.blockBytes.write( b );
	    b = readByte( true );
	  }
	  if( this.blockBytes.size() > 16 ) {
	    this.logBuf.append(
		"Kein Kopfblock gefunden, nur Datenblock gelesen\n" );
	    status = true;
	  } else {
	    this.blockBytes.reset();
	    status = false;
	  }
	  break;
	}
      } else {
	if( b != 0xFF ) {
	  this.fileErr = true;
	  this.logBuf.append(
		String.format(
			"Datenblock hat Blocktyp %02Xh statt FFh\n",
			b ) );
	}
      }
      this.blockBytes.write( b );
      int checkSum = b;

      // Blockinhalt lesen
      int idx = 1;
      while( this.audioThread.isIOEnabled() && (blockLen > 0) ) {
	--blockLen;
	b = readByte( false );
	this.blockBytes.write( b );
	if( headerBlock ) {
	  if( idx < this.headerBytes.length ) {
	    this.headerBytes[ idx++ ] = (byte) b;
	  }
	}
	checkSum = (checkSum ^ b) & 0xFF;
      }

      // Pruefbyte lesen
      if( !this.audioThread.isIOEnabled() ) {
	break;
      }
      b = readByte( false );
      this.blockBytes.write( b );
      if( b != checkSum ) {
	this.blockCSErr = true;
	this.fileErr    = true;
	this.logBuf.append( headerBlock ? "Kopfblock" : "Datenblock" );
	appendColonChecksumErrorTo( this.logBuf );
	this.logBuf.append( '\n' );
      }

      // Block gelesen
      status = true;
      break;
    }
    if( status ) {
      this.audioThread.setBlockEnd( this.blockCSErr );
    }
    return status;
  }


  private int readByte( boolean recognizeBlockEnd )
  {
    int value      = 0;
    int waveMicros = 0;
    for( int i = 0; this.audioThread.isIOEnabled() && (i < 8); i++ ) {
      waveMicros = this.audioThread.readMicrosTillPhaseChange()
			+ this.audioThread.readMicrosTillPhaseChange();
      if( recognizeBlockEnd && (waveMicros >= (2 * BIT_DISTINCT_MICROS)) ) {
	value = -1;
	break;
      }
      value <<= 1;
      if( waveMicros < BIT_DISTINCT_MICROS ) {
	// 0-Bit
	this.audioThread.fillUpRecognizedWave( WAVE_BIT_0 );
      } else {
	// 1-Bit
	value |= 0x01;
	this.audioThread.fillUpRecognizedWave( WAVE_BIT_1 );
      }
    }
    return value;
  }


  private void writeBlock() throws IOException
  {
    int size = this.blockBytes.size();
    if( size > 0 ) {

      // TAP
      this.tapBytes.write( size );
      this.tapBytes.write( size >> 8 );
      this.blockBytes.writeTo( this.tapBytes );

      // TZX
      if( this.tzxBytes.size() == 0 ) {
	String signature = "ZXTape!";
	int    len       = signature.length();
	for( int i = 0; i < len; i++ ) {
	  this.tzxBytes.write( signature.charAt( i ) );
	}
	this.tzxBytes.write( 0x1A );
	this.tzxBytes.write( 1 );	// TZX Hauptversion
	this.tzxBytes.write( 0 );	// TZX Unterversion
      }
      this.tzxBytes.write( 0x10 );	// Standard speed data block
      this.tzxBytes.write( 0xE8 );	// LOW(1000)
      this.tzxBytes.write( 0x03 );	// HIGHT(1000)
      this.tzxBytes.write( size );
      this.tzxBytes.write( size >> 8 );
      this.blockBytes.writeTo( this.tzxBytes );
    }
  }
}
