/*
 * (c) 2023-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Verarbeitung der Audiodaten im Z1013-Format
 */

package jkcload.audio;

import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


public class Z1013AudioProcessor extends FM16x16AudioProcessor
{
  private static final String Z1013_CLASSIC    = "Z1013 classic";
  private static final String Z1013_HEADERSAVE = "Z1013 Headersave";

  private ByteArrayOutputStream fileBytes;
  private StringBuilder         hsLogBuf;
  private StringBuilder         plainLogBuf;
  private boolean               fileErr;
  private int                   filePos;
  private int                   firstBlockAddr;


  public Z1013AudioProcessor(
			AudioThread audioThread,
			float       tolerance,
			int         mhz )
  {
    super(
	audioThread,
	tolerance,
	1540 / mhz,
	770 / mhz,
	572 / mhz );
    this.fileBytes      = new ByteArrayOutputStream( 0x4000 );
    this.hsLogBuf       = new StringBuilder( 1024 );
    this.plainLogBuf    = new StringBuilder( 1024 );
    this.filePos        = 0;
    this.fileErr        = false;
    this.firstBlockAddr = -1;
  }


	/* --- ueberschrieben Methoden --- */

  @Override
  public void checksumError( int blockAddr )
  {
    this.fileErr = true;
    appendCSErrTo( this.hsLogBuf, blockAddr );
    appendCSErrTo( this.plainLogBuf, this.filePos );
  }


  @Override
  public boolean readBlock()
  {
    boolean status = super.readBlock();
    if( status ) {
      this.fileBytes.write( this.blockBuf, 0, this.blockBuf.length );
      this.filePos += 0x20;
    }
    return status;
  }


  @Override
  public void run()
  {
    try {
      this.fileBytes.reset();
      this.hsLogBuf.setLength( 0 );
      this.plainLogBuf.setLength( 0 );
      this.fileErr        = false;
      this.filePos        = 0;
      this.firstBlockAddr = -1;

      if( readBlock() ) {

	// Test auf Headersave
	boolean headersave = false;
	if( (this.blockAddr > 0)
	    && (this.blockBuf[ 13 ] == (byte) 0xD3)
	    && (this.blockBuf[ 14 ] == (byte) 0xD3)
	    && (this.blockBuf[ 15 ] == (byte) 0xD3) )
	{
	  headersave              = true;
	  int           begAddr   = getWord( this.blockBuf, 0 );
	  int           endAddr   = getWord( this.blockBuf, 2 );
	  int           startAddr = getWord( this.blockBuf, 4 );
	  int           fType     = this.blockBuf[ 12 ] & 0xFF;
	  StringBuilder nameBuf   = new StringBuilder( 16 );
	  for( int i = 16; i < 32; i++ ) {
	    int b = this.blockBuf[ i ] & 0xFF;
	    if( (b >= 0x20) && (b < 0x7F) ) {
	      nameBuf.append( (char) b );
	    }
	  }
	  if( (fType != 'C') && (fType != 'M') ) {
	    startAddr = -1;
	  }
	  String fileName = nameBuf.toString().trim();
	  String infoText = null;
	  if( (fType > 0x20) && (fType < 0x7F) ) {
	    infoText = "Dateityp " + String.valueOf( (char) fType );
	  }

	  // Aktivitaetsanzeige
	  StringBuilder activityBuf = new StringBuilder( 64 );
	  activityBuf.append( Z1013_HEADERSAVE );
	  activityBuf.append( ": " );
	  activityBuf.append( fileName );
	  String activity = activityBuf.toString();
	  this.observer.updReadActivity( activity );

	  // Datenbereich der Datei einlesen
	  int addr      = begAddr;
	  int blockAddr = begAddr;
	  while( addr <= endAddr ) {
	    if( !readBlock() ) {
	      break;
	    }
	    this.observer.updReadActivity(
		String.format( "%s %04Xh", activity, this.blockAddr ) );
	    if( this.blockAddr != blockAddr ) {
	      this.hsLogBuf.append(
		  String.format(
			"Block %04Xh gelesen, erwartet %04Xh\n",
			this.blockAddr,
			blockAddr ) );
	      blockAddr    = this.blockAddr;
	      this.fileErr = true;
	    }
	    addr += 0x20;
	    blockAddr += 0x20;
	  }
	  if( addr < endAddr ) {
	    appendIncompleteReadTo( this.hsLogBuf, endAddr - addr + 1 );
	    this.fileErr = true;
	  }
	  List<String> fileExtList = new ArrayList<>();
	  fileExtList.add( "Z80" );
	  byte[]             fileBytes     = this.fileBytes.toByteArray();
	  Map<String,byte[]> fileExt2Bytes = new HashMap<>();
	  fileExt2Bytes.put( "Z80", fileBytes );
	  if( fileBytes.length > 0x20 ) {
	    int    binLen   = fileBytes.length - 0x20;
	    byte[] binBytes = new byte[ binLen ];
	    System.arraycopy( fileBytes, 0x20, binBytes, 0, binLen );
	    fileExtList.add( "BIN" );
	    fileExt2Bytes.put( "BIN", binBytes );
	  }
	  this.observer.fileRead(
				Z1013_HEADERSAVE,
				begAddr,
				addr - begAddr,
				startAddr,
				fileName,
				infoText,
				fileExtList,
				fileExt2Bytes,
				this.hsLogBuf.toString(),
				this.fileErr );
	}
	if( !headersave ) {
	  while( readBlock() ) {
	    this.observer.updReadActivity(
		String.format( "%s: %04Xh", Z1013_CLASSIC, this.filePos ) );
	  }
	  this.observer.fileRead(
			Z1013_CLASSIC,
			this.firstBlockAddr,
			this.fileBytes.size(),
			-1,
			null,
			null,
			Collections.singletonList( "BIN" ),
			Collections.singletonMap(
					"BIN",
					this.fileBytes.toByteArray() ),
			this.plainLogBuf.toString(),
			this.fileErr );
	}
	this.observer.updReadActivity( null );
      }
    }
    catch( Exception ex ) {
      this.observer.errorOccured( null, ex );
    }
  }


	/* --- private Methoden --- */

  private void appendCSErrTo( StringBuilder buf, int addr )
  {
    buf.append( String.format( "Block %04Xh", addr ) );
    appendColonChecksumErrorTo( buf );
    buf.append( '\n' );
  }
}
