/*
 * (c) 2024-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Verarbeitung der Audiodaten im SCCH-Format (AC1/LLC2 TurboSave)
 */

package jkcload.audio;

import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


public class SCCHAudioProcessor extends FM16x16AudioProcessor
{
  private static final String SCCH_NORMAL = "AC1/SCCH-TurboSave";
  private static final String SCCH_RAW    = "AC1/SCCH-TurboSave Rohdaten";

  private ByteArrayOutputStream fileBytes;
  private StringBuilder         logBuf;
  private boolean               fileErr;
  private int                   firstBlockAddr;


  public SCCHAudioProcessor(
			AudioThread audioThread,
			float       tolerance )
  {
    super(
	audioThread,
	tolerance,
	510,
	255,
	192 );
    this.fileBytes = new ByteArrayOutputStream( 0x4000 );
    this.logBuf    = new StringBuilder( 1024 );
    this.fileErr   = false;
  }


	/* --- ueberschrieben Methoden --- */

  @Override
  protected void checksumError( int blockAddr )
  {
    this.fileErr = true;
    this.logBuf.append( String.format( "Block %04Xh", blockAddr ) );
    appendColonChecksumErrorTo( this.logBuf );
    this.logBuf.append( '\n' );
  }


  @Override
  public void run()
  {
    try {
      this.fileBytes.reset();
      this.logBuf.setLength( 0 );
      this.fileErr        = false;
      this.firstBlockAddr = -1;

      if( readBlock() ) {

	// Test auf SCCH-Kennung
	boolean scch = true;
	for( int i = 0; i < 8; i++ ) {
	  if( this.blockBuf[ i ] != (byte) 'N' ) {
	    scch = false;
	    break;
	  }
	}
	if( (this.blockBuf[ 24 ] != (byte) ':')
	    || (this.blockBuf[ 25 ] != (byte) '\u0020') )
	{
	  scch = false;
	}
	if( scch ) {
	  int           begAddr = getWord( this.blockBuf, 29 );
	  int           endAddr = getWord( this.blockBuf, 27 );
	  int           fType   = this.blockBuf[ 26 ] & 0xFF;
	  StringBuilder nameBuf = new StringBuilder( 16 );
	  for( int i = 8; i < 24; i++ ) {
	    int b = this.blockBuf[ i ] & 0xFF;
	    if( (b >= 0x20) && (b < 0x7F) ) {
	      nameBuf.append( (char) b );
	    }
	  }
	  String fileName = nameBuf.toString().trim();
	  String infoText = null;
	  if( (fType > 0x20) && (fType < 0x7F) ) {
	    infoText = "Dateityp " + String.valueOf( (char) fType );
	  }

	  // Aktivitaetsanzeige
	  StringBuilder activityBuf = new StringBuilder( 64 );
	  activityBuf.append( SCCH_NORMAL );
	  activityBuf.append( ": " );
	  activityBuf.append( fileName );
	  String activity = activityBuf.toString();
	  this.observer.updReadActivity( activity );

	  // Datenbereich der Datei einlesen
	  int addr      = begAddr;
	  int blockAddr = begAddr;
	  while( addr <= endAddr ) {
	    if( !readBlock() ) {
	      break;
	    }
	    this.fileBytes.write( this.blockBuf, 0, this.blockBuf.length );
	    this.observer.updReadActivity(
		String.format( "%s %04Xh", activity, this.blockAddr ) );
	    if( this.blockAddr != blockAddr ) {
	      this.logBuf.append(
			String.format(
				"Block %04Xh gelesen, erwartet %04Xh\n",
				this.blockAddr,
				addr ) );
	      blockAddr    = this.blockAddr;
	      this.fileErr = true;
	    }
	    addr += 0x20;
	    blockAddr += 0x20;
	  }
	  if( addr < endAddr ) {
	    appendIncompleteReadTo( this.logBuf, endAddr - addr + 1 );
	    this.fileErr = true;
	  }
	  List<String>       fileExtList   = new ArrayList<>();
	  Map<String,byte[]> fileExt2Bytes = new HashMap<>();
	  byte[]             binBytes      = this.fileBytes.toByteArray();

	  ByteArrayOutputStream z80Bytes = new ByteArrayOutputStream(
						binBytes.length + 0x20 );
	  writeHeadersaveHeader(
			z80Bytes,
			begAddr,
			endAddr,
			0xFFFF,
			fType,
			fileName );
	  z80Bytes.write( binBytes );
	  fileExtList.add( "Z80" );
	  fileExt2Bytes.put( "Z80", z80Bytes.toByteArray() );

	  String fileExt = "BIN";
	  if( fType == 'B' ) {
	    fileExt = "BAS";
	  }
	  fileExtList.add( fileExt );
	  fileExt2Bytes.put( fileExt, binBytes );
	  this.observer.fileRead(
			SCCH_NORMAL,
			begAddr,
			addr - begAddr,
			-1,
			fileName,
			infoText,
			fileExtList,
			fileExt2Bytes,
			this.logBuf.toString(),
			this.fileErr );
	} else {
	  this.fileBytes.write( this.blockBuf, 0, this.blockBuf.length );

	  // Aktivitaetsanzeige
	  this.observer.updReadActivity(
		String.format( "%s %04Xh", SCCH_RAW, this.blockAddr ) );

	  // Datenbereich der Datei einlesen
	  int firstBlockAddr = this.blockAddr;
	  while( readBlock() ) {
	    this.fileBytes.write( this.blockBuf, 0, this.blockBuf.length );
	    this.observer.updReadActivity(
		String.format( "%s %04Xh", SCCH_RAW, this.blockAddr ) );
	  }
	  this.observer.fileRead(
			SCCH_RAW,
			firstBlockAddr,
			this.fileBytes.size(),
			-1,
			null,
			"Rohdaten der physischen Ebene",
			Collections.singletonList( "RAW" ),
			Collections.singletonMap(
					"RAW",
					this.fileBytes.toByteArray() ),
			this.logBuf.toString(),
			this.fileErr );
	}
	this.observer.updReadActivity( null );
      }
    }
    catch( Exception ex ) {
      this.observer.errorOccured( null, ex );
    }
  }
}
