/*
 * (c) 2024-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Basisklasse zur Verarbeitung von phasenmodulierten Audiodaten
 */

package jkcload.audio;

import java.util.Arrays;


public abstract class PhaseModAudioProcessor extends AudioProcessor
{
  private int     halfBitMicros;
  private int     defaultDelimMicros;
  private int     delimMicros;
  private int     pilotMinMicros;
  private int     pilotMaxMicros;
  private boolean bitValue;
  private boolean halfBit;
  private boolean syncDone;
  private boolean negate;


  protected PhaseModAudioProcessor(
			AudioThread audioThread,
			float       tolerance,
			int         pilotMicros,
			int         halfBitMicros )
  {
    super( audioThread );
    this.halfBitMicros      = halfBitMicros;
    this.pilotMinMicros     = Math.round( pilotMicros * (1F - tolerance) );
    this.pilotMaxMicros     = Math.round( pilotMicros * (1F + tolerance) );
    this.defaultDelimMicros = this.halfBitMicros + (this.halfBitMicros / 2);
    reset();
  }


  protected boolean readBit()
  {
    boolean value = this.negate ^ this.bitValue;
    while( this.audioThread.isIOEnabled() ) {
      int m = this.audioThread.readMicrosTillPhaseChange();
      if( !this.syncDone ) {
	if( (m >= this.pilotMinMicros) && (m <= this.pilotMaxMicros) ) {
	  this.audioThread.fillUpRecognizedWave( WAVE_PILOT );
	} else {
	  this.audioThread.fillUpRecognizedWave( NONE );
	}
      }
      if( m > this.delimMicros ) {
	this.bitValue = !this.bitValue;
	break;
      }
      this.halfBit = !this.halfBit;
      if( !this.halfBit ) {
	break;
      }
    }
    if( this.syncDone ) {
      this.audioThread.fillUpRecognizedWave(
				value ? WAVE_BIT_1 : WAVE_BIT_0 );
    }
    return value;
  }


  protected int readByte()
  {
    int b = 0;
    int n = 8;
    while( this.audioThread.isIOEnabled() && (n > 0) ) {
      b <<= 1;
      if( readBit() ) {
	b |= 0x01;
      }
      b &= 0xFF;
      --n;
    }
    if( n > 0 ) {
      b = -1;
    }
    return b;
  }


  protected void reset()
  {
    this.delimMicros = this.defaultDelimMicros;
    this.bitValue    = false;
    this.halfBit     = false;
    this.syncDone    = false;
    this.negate      = false;
  }


  protected boolean waitForSyncByte()
  {
    boolean status     = false;
    int[]   recIndexes = new int[ 9 ];
    Arrays.fill( recIndexes, -1 );
    reset();
    int b = 0;
    while( this.audioThread.isIOEnabled() ) {
      for( int i = recIndexes.length - 2; i >= 0; --i ) {
	recIndexes[ i + 1 ] = recIndexes[ i ];
      }
      recIndexes[ 0 ] = this.audioThread.getRecognitionIndex();
      b = (b << 1) & 0xFE;
      if( readBit() ) {
	b |= 0x01;
      }
      if( b == 0xE6 ) {		// Synchonistions-Byte
	this.negate = false;
	status      = true;
	break;
      }
      if( b == 0x19 ) {		// negiertes Synchonistions-Byte
	this.negate = true;
	status      = true;
	break;
      }
    }
    if( status ) {
      this.syncDone = true;
      int recIdx    = recIndexes[ recIndexes.length - 1 ];
      if( recIdx >= 0 ) {
	this.audioThread.fillUpRecognizedWave( recIdx, WAVE_DELIM );
      }
    }
    return status;
  }
}
