/*
 * (c) 2023-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Verarbeitung der Audiodaten im KC-Format
 */

package jkcload.audio;

import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


public class KCAudioProcessor extends AudioProcessor
{
  /*
   * Messwerte bei Abtastefrequenz 44,1 kHz:
   *
   * Vorton:
   *   Zeit in Mikrosekunden zwischen zwei Phasenwechsel:
   *     Z9001:         454..476
   *     KC85/3:        476..499
   *     KC85/4:        454..499
   *     Z1013 KCBASIC: 454..499
   *
   *   Es wird der Mittelwert zwischen 457 (Z9001) und 488 (KC85/3)
   *   als Sollwert angenommen: 476
   *
   *
   * Trennschwingung:
   *   Zeit in Mikrosekunden zwischen zwei Phasenwechsel:
   *     Z9001:         862..884
   *     KC85/3:        884..907
   *     KC85/4:        884
   *     Z1013 KCBASIC: 1020..1066
   *
   *   Da der Z1013 hier etwas aus der Reihe tanzt,
   *   wird mit 920 ein Sollwert angenommen,
   *   der bei den KCs noch in der 10%-Toleranz
   *   und beim Z1013 noch in der 15%-Toleranz liegt.
   *
   *
   * 0-Bit:
   *   Zeit in Mikrosekunden zwischen zwei Phasenwechsel:
   *     Z9001:         227..249
   *     KC85/3:        272
   *     KC85/4:        249..272
   *     Z1013 KCBASIC: 204..272
   *
   * 1-Bit:
   *   Zeit in Mikrosekunden zwischen zwei Phasenwechsel:
   *     Z9001:         454..476
   *     KC85/3:        476..499
   *     KC85/4:        454..476
   *     Z1013 KCBASIC: 454..522
   *
   * Die Trennung zwischen 0- und 1-Bit wird in der Mitte
   * zwischen dem hoechsten Wert des 0-Bits (272)
   * und dem niedrigsten Wert des 1-Bits (454) festgelegt: 363
   */
  private static final int PILOT_MICROS        = 476;
  private static final int DELIM_MICROS        = 920;
  private static final int BIT_DISTINCT_MICROS = 363;

  private static final String KCTAP_MAGIC = "\u00C3KC-TAPE by AF.\u0020";

  private ByteArrayOutputStream dataBytes;
  private ByteArrayOutputStream tapBytes;
  private StringBuilder         logBuf;
  private byte[]                blockBuf;
  private int                   blockNum;
  private boolean               blockLongPilot;
  private boolean               blockCSErr;
  private boolean               multiTAP;
  private boolean               fileErr;
  private int                   fileBlocks;
  private int                   delimMinMicros;
  private int                   delimMaxMicros;
  private int                   pilotMinMicros;
  private int                   pilotMaxMicros;
  private String                dataFileExt;


  public KCAudioProcessor( AudioThread audioThread, float tolerance )
  {
    super( audioThread );
    this.dataBytes      = new ByteArrayOutputStream( 0x4000 );
    this.tapBytes       = new ByteArrayOutputStream( 0x4000 );
    this.logBuf         = new StringBuilder( 1024 );
    this.blockBuf       = new byte[ 128 ];
    this.blockNum       = 0;
    this.blockLongPilot = false;
    this.blockCSErr     = false;
    this.multiTAP       = false;
    this.fileErr        = false;
    this.fileBlocks     = 0;
    this.dataFileExt    = null;

    // Zeiten berechnen
    int absTolerance    = Math.round( (float) PILOT_MICROS * tolerance );
    this.pilotMinMicros = PILOT_MICROS - absTolerance;
    this.pilotMaxMicros = PILOT_MICROS + absTolerance;
    absTolerance        = Math.round( (float) DELIM_MICROS * tolerance );
    this.delimMinMicros = DELIM_MICROS - absTolerance;
    this.delimMaxMicros = DELIM_MICROS + absTolerance;
 }


	/* --- ueberschrieben Methoden --- */

  @Override
  public void run()
  {
    try {
      this.dataBytes.reset();
      this.tapBytes.reset();
      this.logBuf.setLength( 0 );
      this.blockNum       = 0;
      this.blockLongPilot = false;
      this.blockCSErr     = false;
      this.multiTAP       = false;
      this.fileErr        = false;
      this.fileBlocks     = 0;
      this.dataFileExt    = null;

      StringBuilder activityBuf = new StringBuilder( 64 );
      String        format      = null;
      String        fileName    = null;
      String        fileType    = null;
      int           begAddr     = -1;
      int           startAddr   = -1;
      int           blockNum    = this.blockNum;
      boolean       blockStatus = false;
      boolean       analyzing   = true;
      for(;;) {
	if( !readBlock() ) {
	  break;
	}
	writeBlock();
	if( analyzing ) {
	  if( this.blockNum == 0 ) {
	    format   = "KC:Z9001";
	    fileName = getText( 0, 8 );
	    fileType = getText( 8, 11 );
	    this.dataFileExt = "KCC";
	    this.dataBytes.write( this.blockBuf );
	  } else if( this.blockNum == 1 ) {
	    boolean kcbasic  = true;
	    int     typeByte = this.blockBuf[ 0 ] & 0xFF;
	    for( int i = 1; i < 2; i++ ) {
	      if( (this.blockBuf[ i ] & 0xFF) != typeByte ) {
		kcbasic = false;
		break;
	      }
	    }
	    if( kcbasic ) {
	      switch( typeByte ) {
		case 0xD3:
		case 0xD7:
		  format           = "KCBASIC-Programm";
		  this.dataFileExt = "SSS";
		  break;

		case 0xD4:
		case 0xD8:
		  format           = "KCBASIC-Datenfeld";
		  this.dataFileExt = "TTT";
		  break;

		case 0xD5:
		case 0xD9:
		  format           = "KCBASIC-Listing";
		  this.dataFileExt = "UUU";
		  break;

		default:
		  kcbasic = false;
	      }
	      if( kcbasic ) {
		if( (typeByte >= 0xD7) && (typeByte <= 0xD9) ) {
		  format += " mit Kopierschutz";
		}
		fileName = getText( 3, 11 );
		fileType = null;
		this.dataBytes.write(
				this.blockBuf,
				11,
				this.blockBuf.length - 11 );
	      }
	    }
	    if( !kcbasic ) {
	      if( format == null ) {
		format   = "KC85/2..5";
		fileName = getText( 0, 11 );
		fileType = null;
		begAddr  = getWord( this.blockBuf, 17 );
		if( this.blockBuf[ 16 ] == (byte) 3 ) {
		  startAddr = getWord( this.blockBuf, 21 );
		}
		this.dataFileExt = "KCC";
		this.dataBytes.write( this.blockBuf );
	      }
	    }
	    analyzing = false;
	  } else {
	    if( format == null ) {
	      format   = "KC";
	      fileName = null;
	      fileType = null;
	    }
	    analyzing = false;
	  }
	} else {
	  if( this.dataFileExt != null ) {
	    this.dataBytes.write( this.blockBuf );
	  }
	}
	if( this.blockLongPilot ) {
	  blockNum = this.blockNum;
	} else {
	  blockNum = (blockNum + 1) & 0xFF;
	  if( (blockNum != this.blockNum) && (this.blockNum != 0xFF) ) {
	    this.logBuf.append(
		String.format( "  Block %02Xh erwartet\n", blockNum ) );
	    this.fileErr = true;
	  }
	}
	activityBuf.setLength( 0 );
	activityBuf.append( format );
	activityBuf.append( ": " );
	if( fileName != null ) {
	  if( !fileName.isEmpty() ) {
	    activityBuf.append( fileName );
	    if( fileType != null ) {
	      if( !fileType.isEmpty() ) {
		activityBuf.append( '.' );
		activityBuf.append( fileType );
	      }
	    }
	  }
	}
	activityBuf.append( String.format( " %02Xh", this.blockNum ) );
	if( this.blockCSErr ) {
	  activityBuf.append( " fehlerhaft gelesen" );
	}
	this.observer.updReadActivity( activityBuf.toString() );
      }
      if( this.tapBytes.size() > 0 ) {
	List<String> fileExtList = new ArrayList<>();
	fileExtList.add( "TAP" );
	Map<String,byte[]> fileExt2Bytes = new HashMap<>();
	fileExt2Bytes.put( "TAP", this.tapBytes.toByteArray() );
	if( !this.multiTAP && (this.dataFileExt != null) ) {
	  int size = this.dataBytes.size();
	  int nBlk = size / 0x80;
	  int nRemain = (nBlk * 0x80) - size;
	  while( nRemain > 0 ) {
	    this.dataBytes.write( 0 );
	    --nRemain;
	  }
	  fileExtList.add( this.dataFileExt );
	  fileExt2Bytes.put(
			this.dataFileExt,
			this.dataBytes.toByteArray() );
	}
	String infoText = null;
	if( this.multiTAP ) {
	  infoText = "Multi-TAP";
	} else if( fileType != null ) {
	  if( fileType.isEmpty() ) {
	    infoText = "Dateityp: " + fileType;
	  }
	}
	this.observer.fileRead(
			format,
			begAddr,
			this.fileBlocks * 0x80,
			startAddr,
			fileName,
			infoText,
			fileExtList,
			fileExt2Bytes,
			this.logBuf.toString(),
			this.fileErr );
      }
    }
    catch( Exception ex ) {
      this.observer.errorOccured( null, ex );
    }
  }


	/* --- private Methoden --- */

  private String getText( int idx1, int idx2 )
  {
    StringBuilder buf = new StringBuilder();
    while( idx1 < idx2 ) {
      int b = this.blockBuf[ idx1++ ] & 0xFF;
      if( (b < 0x20) || (b > 0x7E) ) {
	break;
      }
      buf.append( (char) b );
    }
    return buf.toString();
  }


  private boolean matchesDelimMicros( int micros )
  {
    return (micros >= this.delimMinMicros)
	   && (micros <= this.delimMaxMicros);
  }


  private boolean matchesPilotMicros( int micros )
  {
    return (micros >= this.pilotMinMicros)
	   && (micros <= this.pilotMaxMicros);
  }


  private boolean readBlock()
  {
    this.blockNum       = -1;
    this.blockLongPilot = false;
    this.blockCSErr     = false;

    boolean status           = false;
    long    microsSinceStart = 0;
    while( this.audioThread.isIOEnabled() ) {
      this.audioThread.fillUpRecognizedWave( NONE );

      /*
       * Zeit seit Start der Blocksuche pruefen,
       * nach 0,3 Sekunden abbrechen
       */
      if( microsSinceStart > 300000 ) {
	break;
      }

      // Vorton finden
      int micros = this.audioThread.readMicrosTillPhaseChange();
      microsSinceStart += micros;
      if( !matchesPilotMicros( micros ) ) {
	continue;
      }

      // Vorton uebergehen
      int  nPilotPulses = 0;
      while( this.audioThread.isIOEnabled() ) {
	this.audioThread.markBlockBegin();
	this.audioThread.fillUpRecognizedWave( WAVE_PILOT );
	micros = this.audioThread.readMicrosTillPhaseChange();
	microsSinceStart += micros;
	if( !matchesPilotMicros( micros ) ) {
	  break;
	}
	nPilotPulses++;
      }
      this.blockLongPilot = (nPilotPulses > 500);

      // Trennschwingung
      if( !matchesDelimMicros( micros ) ) {
	continue;
      }
      this.audioThread.fillUpRecognizedWave( WAVE_DELIM );
      micros = this.audioThread.readMicrosTillPhaseChange();
      microsSinceStart += micros;
      if( !matchesDelimMicros( micros ) ) {
	continue;
      }
      this.audioThread.fillUpRecognizedWave( WAVE_DELIM );

      // Blocknummer
      this.blockNum = readByte();

      // 128 Datenbytes lesen
      int checksum = 0;
      int idx      = 0;
      while( this.audioThread.isIOEnabled()
	     && (idx < this.blockBuf.length) )
      {
	int b    = readByte();
	checksum = (checksum + b) & 0xFF;
	this.blockBuf[ idx++ ] = (byte) b;
      }
      this.logBuf.append( String.format( "Block %02Xh", this.blockNum ) );
      if( idx < this.blockBuf.length ) {
	this.logBuf.append( ": " );
	appendIncompleteReadTo( this.logBuf, this.blockBuf.length - idx );
	while( idx < this.blockBuf.length ) {
	  this.blockBuf[ idx++ ] = (byte) 0;
	}
      } else {

	// Pruefsumme
	if( readByte() != checksum ) {
	  this.blockCSErr = true;
	  this.fileErr    = true;
	  appendColonChecksumErrorTo( this.logBuf );
	}
      }
      this.logBuf.append( '\n' );

      // Block gelesen
      this.audioThread.setBlockEnd( this.blockCSErr );
      status = true;
      break;
    }
    return status;
  }


  private int readByte()
  {
    int v = 0;
    for( int i = 0; this.audioThread.isIOEnabled() && (i < 8); i++ ) {
      v >>= 1;
      if( (this.audioThread.readMicrosTillPhaseChange()
			+ this.audioThread.readMicrosTillPhaseChange())
		< (2 * BIT_DISTINCT_MICROS) )
      {
	// 0-Bit
	this.audioThread.fillUpRecognizedWave( WAVE_BIT_0 );
      } else {
	// 1-Bit
	v |= 0x80;
	this.audioThread.fillUpRecognizedWave( WAVE_BIT_1 );
      }
    }
    this.audioThread.readMicrosTillPhaseChange();
    this.audioThread.readMicrosTillPhaseChange();
    this.audioThread.fillUpRecognizedWave( WAVE_DELIM );
    return v;
  }


  private void writeBlock()
  {
    if( this.tapBytes.size() > 0 ) {
      if( this.blockLongPilot ) {
	writeKCTapHeader();
	this.multiTAP = true;
      }
    } else {
      writeKCTapHeader();
    }
    this.tapBytes.write( this.blockNum );
    this.tapBytes.write( this.blockBuf, 0, this.blockBuf.length );
    this.fileBlocks++;
  }


  private void writeKCTapHeader()
  {
    int len = KCTAP_MAGIC.length();
    for( int i = 0; i < len; i++ ) {
      this.tapBytes.write( KCTAP_MAGIC.charAt( i ) );
    }
  }
}
