/*
 * (c) 2024-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Verarbeitung von frequenzmodulierten Audiodaten,
 * die folgenden Blockaufbau haben:
 *   Vorton
 *   Trennschwingung
 *    1 16-Bit-Datenwort Blockadresse
 *   16 16-Bit-Datenwoerter Nutzdaten
 *    1 16-Bit-Datenwort Pruefsumme
 *
 *   0-Bit: volle Schwingung mit doppelter Frequenz wie 1-Bit
 *   1-Bit: Halbschwingung
 */

package jkcload.audio;


public abstract class FM16x16AudioProcessor extends AudioProcessor
{
  protected byte[] blockBuf;
  protected int    blockAddr;

  private boolean blockCSErr;
  private int     bitDistinctMicros;
  private int     delimMinMicros;
  private int     delimMaxMicros;
  private int     pilotMinMicros;
  private int     pilotMaxMicros;


  protected FM16x16AudioProcessor(
			AudioThread audioThread,
			float       tolerance,
			int         pilotMicros,
			int         delimMicros,
			int         bitDistinctMicros )
  {
    super( audioThread );
    this.blockBuf    = new byte[ 32 ];
    this.blockAddr   = 0;
    this.blockCSErr  = false;

    // Zeiten berechnen
    int absTolerance       = Math.round( (float) pilotMicros * tolerance );
    this.pilotMinMicros    = pilotMicros - absTolerance;
    this.pilotMaxMicros    = pilotMicros + absTolerance;
    absTolerance           = Math.round( (float) delimMicros * tolerance );
    this.delimMinMicros    = delimMicros - absTolerance;
    this.delimMaxMicros    = delimMicros + absTolerance;
    this.bitDistinctMicros = bitDistinctMicros;
  }


  protected void checksumError( int blockAddr )
  {
    // zu ueberschreiben
  }


  protected boolean readBlock()
  {
    this.blockCSErr = false;
    this.blockAddr  = 0;

    boolean status           = false;
    long    microsSinceStart = 0;
    int[]   wordBuf          = new int[ 18 ];
    while( this.audioThread.isIOEnabled() ) {
      this.audioThread.fillUpRecognizedWave( NONE );

      /*
       * Wenn 0,5 Sekunden nach Start der Blocksuche
       * der Block nicht gelesen werden konnte,
       * wird abgebrochen.
       */
      if( microsSinceStart > 500000 ) {
	break;
      }

      // Vorton finden
      int micros = this.audioThread.readMicrosTillPhaseChange();
      microsSinceStart += micros;
      if( !matchesPilotMicros( micros ) ) {
	continue;
      }

      // Vorton uebergehen
      this.audioThread.markBlockBegin();
      int nPilotPulses = 0;
      while( this.audioThread.isIOEnabled() ) {
	this.audioThread.fillUpRecognizedWave( WAVE_PILOT );
	micros = this.audioThread.readMicrosTillPhaseChange();
	microsSinceStart += micros;
	if( !matchesPilotMicros( micros ) ) {
	  if( nPilotPulses > 4 ) {
	    // wahrscheinlich Fehler im Vorton -> Timeout neu beginnen
	    microsSinceStart = 0;
	  }
	  break;
	}
	nPilotPulses++;
      }

      // Trennschwingung
      if( !matchesDelimMicros( micros ) ) {
	continue;
      }
      this.audioThread.fillUpRecognizedWave( WAVE_DELIM );
      micros = this.audioThread.readMicrosTillPhaseChange();
      microsSinceStart += micros;
      if( !matchesDelimMicros( micros ) ) {
	continue;
      }
      this.audioThread.fillUpRecognizedWave( WAVE_DELIM );

      // 18 Woerter lesen
      int idx = 0;
      while( this.audioThread.isIOEnabled() && (idx < wordBuf.length) ) {
	wordBuf[ idx++ ] = readWord();
      }
      if( idx == wordBuf.length ) {

	// Pruefsumme ermitteln und vergleichen
	int checksum = 0;
	for( int i = 0; i < 17; i++ ) {
	  checksum = (checksum + wordBuf[ i ]) & 0xFFFF;
	}
	this.blockCSErr = ((checksum & 0xFFFF) != wordBuf[ 17 ]);

	// Blockadresse extrahieren und Fehlerprotokolle fortschreiben
	this.blockAddr = wordBuf[ 0 ];
	if( this.blockCSErr ) {
	  checksumError( this.blockAddr );
	}

	// Datenbytes extrahieren
	idx = 0;
	for( int i = 1; i < 17; i++ ) {
	  int b = wordBuf[ i ] & 0xFF;
	  this.blockBuf[ idx++ ] = (byte) b;
	  b = (wordBuf[ i ] >> 8);
	  this.blockBuf[ idx++ ] = (byte) b;
	}

	// Block gelesen
	this.audioThread.setBlockEnd( this.blockCSErr );
	status = true;
	break;
      }
    }
    return status;
  }


	/* --- private Methoden --- */

  private boolean matchesDelimMicros( int micros )
  {
    return (micros >= this.delimMinMicros)
	   && (micros <= this.delimMaxMicros);
  }


  private boolean matchesPilotMicros( int micros )
  {
    return (micros >= this.pilotMinMicros)
	   && (micros <= this.pilotMaxMicros);
  }


  private int readWord()
  {
    int v = 0;
    for( int i = 0; this.audioThread.isIOEnabled() && (i < 16); i++ ) {
      v >>= 1;
      if( this.audioThread.readMicrosTillPhaseChange()
					< this.bitDistinctMicros )
      {
	// 0-Bit -> noch ein Phasenwechsel
	this.audioThread.readMicrosTillPhaseChange();
	this.audioThread.fillUpRecognizedWave( WAVE_BIT_0 );
      } else {
	// 1-Bit
	v |= 0x8000;
	this.audioThread.fillUpRecognizedWave( WAVE_BIT_1 );
      }
    }
    return v;
  }
}
