/*
 * (c) 2024-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Verarbeitung der Audiodaten im BCS3-Format
 *
 * Dateiaufbau:
 *   pro Datenblock:
 *     n x Kennton AA
 *     Sync-Byte E6
 *     Blocklaenge (00: 256)
 *     H-Byte Blockaddresse
 *     L-Byte Blockaddresse
 *     00
 *     Datenbytes...
 *     negierte Pruefsumme
 *
 *   Endeblock:
 *     16 x AA
 *     E6
 *     00
 *     00
 *     L-Byte Programmlaenge
 *     H-Byte Programmlaenge
 *     AA
 */

package jkcload.audio;

import java.io.ByteArrayOutputStream;
import java.util.Collections;


public class BCS3AudioProcessor extends PhaseModAudioProcessor
{
  private static final int HALF_BIT_2_5MHZ_MICROS = 408;  // halbe Bit-Laenge
  private static final int PILOT_2_5MHZ_MICROS = HALF_BIT_2_5MHZ_MICROS * 2;

  private StringBuilder         logBuf;
  private ByteArrayOutputStream fileBytes;


  public BCS3AudioProcessor(
			AudioThread audioThread,
			float       tolerance,
			boolean     mhz3_5 )
  {
    super(
	audioThread,
	tolerance,
	mhz3_5 ? (HALF_BIT_2_5MHZ_MICROS * 25 / 35) : PILOT_2_5MHZ_MICROS,
	mhz3_5 ? (HALF_BIT_2_5MHZ_MICROS * 25 / 35)
					: HALF_BIT_2_5MHZ_MICROS );

    this.fileBytes = new ByteArrayOutputStream( 0x4000 );
    this.logBuf    = new StringBuilder();
  }


	/* --- ueberschrieben Methoden --- */

  @Override
  public void run()
  {
    try {
      this.audioThread.fillUpRecognizedWave( NONE );
      this.fileBytes.reset();
      this.logBuf.setLength( 0 );
      int     begAddr      = -1;
      int     curAddr      = -1;
      int     fileLen      = -1;
      boolean fileErr      = false;
      boolean activityDone = false;
      String  infoText     = null;

      // Bloecke lesen
      while( this.audioThread.isIOEnabled() ) {
	if( !waitForSyncByte() ) {
	  break;
	}
	this.audioThread.markBlockBegin();

	// Pruefsumme
	int cks = 0;

	// Bloecklaenge
	int blockLen = readByte();
	if( blockLen < 0 ) {
	  break;
	}
	cks += blockLen;
	if( blockLen == 0 ) {
	  blockLen = 0x100;
	}

	/*
	 * Blockadresse,
	 * Wenn H-Teil=0 dann Endeblock
	 */
	int h = readByte();
	if( h < 0 ) {
	  break;
	}
	cks += h;

	int l = readByte();
	if( l < 0 ) {
	  break;
	}
	cks += l;

	if( h > 0 ) {

	  // Aktivitaetsanzeige
	  if( !activityDone ) {
	    this.observer.updReadActivity( "BCS3" );
	    activityDone = true;
	  }

	  // Blockadresse
	  int expectedAddr = -1;
	  int blockAddr    = ((h << 8) & 0xFF00) | (l & 0x00FF);
	  this.observer.updReadActivity(
				String.format( "BCS3: %04Xh", blockAddr ) );
	  this.logBuf.append( String.format( "Block %04Xh", blockAddr ) );
	  if( begAddr < 0 ) {
	    begAddr  = blockAddr;
	    curAddr  = blockAddr;
	    infoText = getInfoByBegAddr( blockAddr );
	  } else if( blockAddr != curAddr ) {
	    expectedAddr = curAddr;
	    curAddr      = blockAddr;
	  }

	  // Nullbyte ueberlesen
	  cks += readByte();

	  // Datenbytes
	  int b = 0;
	  while( blockLen > 0 ) {
	    b = readByte();
	    if( b < 0 ) {
	      break;
	    }
	    cks += b;
	    this.fileBytes.write( b );
	    --blockLen;
	  }

	  // Pruefsumme
	  boolean blockErr = false;
	  b                = readByte();
	  if( ((b + cks) & 0xFF) != 0 ) {
	    blockErr = true;
	    fileErr  = true;
	    appendColonChecksumErrorTo( this.logBuf );
	  }
	  this.logBuf.append( '\n' );
	  if( expectedAddr >= 0 ) {
	    this.logBuf.append(
		String.format( "  Block %04Xh erwartet", expectedAddr ) );
	  }
	  this.audioThread.setBlockEnd( blockErr );

	} else {

	  // Endeblock, L-Laenge schon gelesen
	  h = readByte();
	  if( (l >= 0) && (h >= 0) ) {
	    fileLen = ((h << 8) & 0xFF00) | (l & 0x00FF);
	    this.logBuf.append(
			String.format(
				"Endeblock, Dateil\u00E4nge: %04Xh\n",
				fileLen ) );
	  }
	  this.audioThread.setBlockEnd( false );

	  // AA-Byte ueberlesen
	  readByte();
	  break;
	}
      }

      // Dateilaenge pruefen
      if( this.fileBytes.size() > 0 ) {
	String errMsg   = null;
	int    nMissing = fileLen - this.fileBytes.size();
	if( (fileLen < 0) || (nMissing > 0) ) {
	  fileErr = true;
	  appendIncompleteReadTo( this.logBuf, nMissing );
	}
	this.observer.fileRead(
			"BCS3",
			begAddr,
			this.fileBytes.size(),
			-1,
			null,
			infoText,
			Collections.singletonList( "BIN" ),
			Collections.singletonMap(
					"BIN",
					this.fileBytes.toByteArray() ),
			this.logBuf.toString(),
			fileErr );
      }
      this.observer.updReadActivity( null );
    }
    catch( Exception ex ) {
      this.observer.errorOccured( null, ex );
    }
  }


	/* --- private Methoden --- */

  /*
   * Anhand der Anfangsadresse laesst sich die BASIC-Version
   * und die Anzahl der sichtbaren Textzeile ermitteln.
   *
   * SE-BASIC 3.1 mit 2,5 MHz:
   *   Anfangsadresse = 3C82h + (Zeilen * 30)
   *
   * SE-BASIC 3.1 mit 3,5 MHz:
   *   Anfangsadresse = 3C82h + (Zeilen * 41)
   *
   * S/P-BASIC 3.3 mit 2,5 MHz:
   *   Anfangsadresse = 3CB6h + (Zeilen * 30)
   */
  private static String getInfoByBegAddr( int begAddr )
  {
    String rv = null;
    if( begAddr >= (0x3C82 + 60) ) {

      // SE-BASIC 3.1 mit 2,5 MHz pruefen
      int lines = (begAddr - 0x3C82) / 30;
      if( (0x3C82 + (lines * 30)) == begAddr ) {
	rv = String.format(
			"SE-BASIC 3.1 mit 2,5 MHz und %d sichtbaren Zeilen",
			lines );
      }

      // SE-BASIC 3.1 mit 3,5 MHz pruefen
      if( rv == null ) {
	lines = (begAddr - 0x3C82) / 41;
	if( (0x3C82 + (lines * 41)) == begAddr ) {
	  rv = String.format(
			"SE-BASIC 3.1 mit 3,5 MHz und %d sichtbaren Zeilen",
			lines );
	}
      }

      // S/P-BASIC 3.3 mit 2,5 MHz pruefen
      if( rv == null ) {
	lines = (begAddr - 0x3CB6) / 30;
	if( (0x3CB6 + (lines * 30)) == begAddr ) {
	  rv = String.format(
			"S/P-BASIC 3.3 mit 2,5 MHz und %d sichtbaren Zeilen",
			lines );
	}
      }
    }
    return rv;
  }
}
