/*
 * (c) 2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Verarbeitung der Audiodaten im BASICODE-Format
 */

package jkcload.audio;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collections;
import jkcload.Main;


public class BCAudioProcessor extends AudioProcessor
{
  public static final String PROP_PREFIX  = Main.PROP_PREFIX + "basicode.";
  public static final String PROP_PRG_EOL = PROP_PREFIX + ".program.eol";
  public static final String PROP_PRG_RM_EMPTY_LINES
			= PROP_PREFIX + ".program.remove_empty_lines";

  public static final String VALUE_CRNL = "crnl";
  public static final String VALUE_NL   = "cr";
  public static final String VALUE_RAW  = "raw";

  /*
   * Pulsewerte (Halbwelle);
   *   Vorton: 208 Mikrosekunden (2400 Hz)
   *   0-Bit:  417 Mikrosekunden (1200 Hz)
   *   1-Bit : 208 Mikrosekunden (2400 Hz)
   */
  private static final int PILOT_MICROS = 208;

  private static final String BC_PROGRAM = "BASICODE Programm";
  private static final String BC_DATA    = "BASICODE Daten";
  private static final String BC_RAW     = "BASICODE Rohdaten";


  // 208 + ((417 - 208) / 2)
  private static final int BIT_DISTINCT_MICROS = 313;

  private ExtByteArrayOutputStream blockBytes;
  private ByteArrayOutputStream    fileBytes;
  private StringBuilder            blockLogBuf;
  private StringBuilder            fileLogBuf;
  private String                   eol;
  private boolean                  blockErr;
  private boolean                  fileErr;
  private boolean                  removeEmptyLines;
  private int                      pilotMinMicros;
  private int                      pilotMaxMicros;
  private int                      blockBegByte;
  private int                      blockNum;


  public BCAudioProcessor( AudioThread audioThread, float tolerance )
  {
    super( audioThread );
    this.blockBytes   = new ExtByteArrayOutputStream( 0x4000 );
    this.blockLogBuf  = new StringBuilder( 1024 );
    this.blockErr     = false;
    this.blockBegByte = 0;
    this.blockNum     = 0x80;
    this.fileBytes    = new ByteArrayOutputStream( 0x4000 );
    this.fileLogBuf   = new StringBuilder( 1024 );
    this.fileErr      = false;

    // Zeiten berechnen
    int absTolerance    = Math.round( (float) PILOT_MICROS * tolerance );
    this.pilotMinMicros = PILOT_MICROS - absTolerance;
    this.pilotMaxMicros = PILOT_MICROS + absTolerance;

    // Zeilenendebytes bei Programmen
    this.eol   = "\r\n";
    String eol = Main.getProperty( PROP_PRG_EOL );
    if( eol != null ) {
      if( eol.equals( VALUE_NL ) ) {
	this.eol = "\n";
      } else if( eol.equals( VALUE_RAW ) ) {
	this.eol = null;
      }
    }

    // Leerzeichen in Programmen entfernen
    String rmEmptyLines = Main.getProperty( PROP_PRG_RM_EMPTY_LINES );
    if( rmEmptyLines != null ) {
      this.removeEmptyLines = Boolean.parseBoolean( rmEmptyLines );
    } else {
      this.removeEmptyLines = false;
    }
 }


	/* --- ueberschrieben Methoden --- */

  @Override
  public void run()
  {
    try {
      this.fileBytes.reset();
      this.fileLogBuf.setLength( 0 );
      this.fileErr  = false;

      int expectedBlockNum = 0x80;
      while( readBlock() ) {
	if( this.blockBegByte != 0x81 ) {
	  flushFile();
	  expectedBlockNum = 0x80;
	  if( this.blockBegByte == 0x82 ) {
	    ByteArrayOutputStream fileBytes = this.blockBytes;
	    if( this.eol != null ) {
	      this.fileBytes.reset();
	      convertLineEnds(
			this.blockBytes.createInputStream(),
			this.fileBytes );
	      fileBytes = this.fileBytes;
	    }
	    this.observer.fileRead(
			BC_PROGRAM,
			-1,
			fileBytes.size(),
			-1,
			null,
			null,
			Collections.singletonList( "ASC" ),
			Collections.singletonMap(
					"ASC",
					fileBytes.toByteArray() ),
			this.blockLogBuf.toString(),
			this.blockErr );
	    this.fileBytes.reset();
	  } else {
	    this.observer.fileRead(
			BC_RAW,
			-1,
			this.blockBytes.size(),
			-1,
			null,
			null,
			Collections.singletonList( "RAW" ),
			Collections.singletonMap(
					"RAW",
					this.blockBytes.toByteArray() ),
			this.blockLogBuf.toString(),
			this.blockErr );
	  }
	  break;
	}
	if( this.blockNum == 0x80 ) {
	  flushFile();
	  expectedBlockNum = 0x80;
	}
	this.fileLogBuf.append(
		String.format( "Block %02Xh\n", this.blockNum ^ 0x80 ) );
	if( this.blockNum != expectedBlockNum ) {
	  this.fileErr = true;
	  this.fileLogBuf.append(
		String.format(
			"Fehler in der Reihenfolge: Block %02Xh erwartet\n",
			expectedBlockNum ^ 0x80 ) );
	}
	this.fileLogBuf.append( this.blockLogBuf );
	this.blockBytes.writeTo( this.fileBytes );
	this.fileErr |= this.blockErr;
	expectedBlockNum = this.blockNum + 1;
      }
      flushFile();
      this.audioThread.fillUpRecognizedWave( NONE );
    }
    catch( Exception ex ) {
      this.observer.errorOccured( null, ex );
    }
  }


	/* --- private Methoden --- */

  private void convertLineEnds( InputStream in, OutputStream out )
  {
    try {
      boolean empty = true;
      boolean cr    = false;
      int     b     = in.read();
      while( b >= 0 ) {
	if( b == '\r' ) {
	  if( cr ) {
	    if( !empty || !this.removeEmptyLines ) {
	      writeEOL( out );
	      empty = true;
	    }
	  }
	  cr = true;
	} else if( b == '\n' ) {
	  if( !empty || !this.removeEmptyLines ) {
	    writeEOL( out );
	    empty = true;
	  }
	  cr = false;
	} else {
	  if( cr ) {
	    if( !empty || !this.removeEmptyLines ) {
	      writeEOL( out );
	      empty = true;
	    }
	    cr = false;
	  }
	  out.write( b );
	  empty = false;
	}
	b = in.read();
      }
      if( cr ) {
	if( !empty || !this.removeEmptyLines ) {
	  writeEOL( out );
	}
      }
    }
    catch( IOException ex ) {}
  }


  private void flushFile()
  {
    int fileLen = this.fileBytes.size();
    if( fileLen > 0 ) {
      this.observer.fileRead(
			BC_DATA,
			-1,
			fileLen,
			-1,
			null,
			null,
			Collections.singletonList( "DAT" ),
			Collections.singletonMap(
					"DAT",
					this.fileBytes.toByteArray() ),
			this.fileLogBuf.toString(),
			this.fileErr );
    }
    this.fileBytes.reset();
    this.fileLogBuf.setLength( 0 );
  }


  private boolean matchesPilotMicros( int micros )
  {
    return (micros >= this.pilotMinMicros)
	   && (micros <= this.pilotMaxMicros);
  }


  /*
   * Einlesen eines Blocks
   *
   * Bit 7 ist immer negiert.
   */
  private boolean readBlock()
  {
    this.blockBytes.reset();
    this.blockLogBuf.setLength( 0 );
    this.blockErr = false;

    boolean status = false;
    while( this.audioThread.isIOEnabled() ) {
      this.audioThread.fillUpRecognizedWave( NONE );

      // Vorton finden
      int micros = this.audioThread.readMicrosTillPhaseChange();
      if( !matchesPilotMicros( micros ) ) {
	continue;
      }

      /*
       * Es muessen mindestens zehn 1-Bits (20 Halbwellen) erkannt werden,
       * um sicherzugehen, dass es der Vorton und nicht ein 1-Bit
       * innerhalb eines Bytes ist.
       */
      boolean failed = false;
      for( int i = 0; i < 20; i++ ) {
	micros = this.audioThread.readMicrosTillPhaseChange();
	if( !matchesPilotMicros( micros ) ) {
	  failed = true;
	  break;
	}
      }
      if( !this.audioThread.isIOEnabled() ) {
	break;
      }
      if( failed ) {
	continue;
      }
      this.audioThread.markBlockBegin();

      // Startbyte lesen
      int b = readByte();
      if( b < 0 ) {
	break;
      }
      boolean raw       = false;
      int     cks       = b;
      this.blockBegByte = b;

      // restliche Bytes lesen
      if( this.blockBegByte == 0x81 ) {		// Datenblock
	this.blockNum = readByte();
	this.observer.updReadActivity(
		String.format(
			"BASICODE Datenblock %02Xh",
			this.blockNum ^ 0x80 ) );
	cks ^= this.blockNum;
	for( int i = 0; i < 1024; i++ ) {
	  b = readByte();
	  if( b < 0 ) {
	    break;
	  }
	  cks ^= b;
	  this.blockBytes.write( b ^ 0x80 );
	}

	/*
	 * Nun muesste das Blockendebyte 0x83 folgen,
	 * doch manche BasCoder schreiben laengere Bloecke.
	 * Aus diesem Grund wird bis zum Blockendebyte weiter gelesen.
	 */
	b = readByte();
	while( (b >= 0) && (b != 0x83) ) {
	  cks ^= b;
	  this.blockBytes.write( b ^ 0x80 );
	  b = readByte();
	}
      } else if( this.blockBegByte == 0x82 ) {	// Programm
	this.observer.updReadActivity( BC_PROGRAM );
	b = readByte();
	while( (b >= 0) && (b != 0x83) ) {
	  cks ^= b;
	  this.blockBytes.write( b ^ 0x80 );
	  b = readByte();
	}
      } else {
	raw = true;
	this.observer.updReadActivity( BC_RAW );
	b = readByte();
	while( (b >= 0) && (b != 0x83) ) {
	  this.blockBytes.write( this.blockBegByte ^ 0x80 );
	  cks ^= b;
	  this.blockBytes.write( b ^ 0x80 );
	  b = readByte();
	}
      }
      if( b < 0 ) {
	this.blockLogBuf.append( "Vorzeitiges Blockende\n" );
	this.blockErr = true;
      } else if( b == 0x83 ) {
	if( raw ) {
	  this.blockBytes.write( b ^ 0x80 );
	}
	cks ^= b;
	b = readByte();
	if( raw ) {
	  this.blockBytes.write( b ^ 0x80 );
	}
	if( b != cks ) {
	  this.blockLogBuf.append( CKS_ERROR );
	  this.blockLogBuf.append( '\n' );
	  this.blockErr = true;
	}
      } else {
	this.blockLogBuf.append( "Blockendebyte nicht gefunden\n" );
	this.blockErr = true;
      }

      /*
       * nachfolgenden Kennton ueberlesen,
       * damit er nicht als Anfang des naechsten Blocks erkannt wird
       */
      while( this.audioThread.isIOEnabled() ) {
	if( !matchesPilotMicros(
			this.audioThread.readMicrosTillPhaseChange() ) )
	{
	  break;
	}
	this.audioThread.fillUpRecognizedWave( WAVE_PILOT );
      }

      // Block gelesen
      status = true;
      break;
    }
    if( status ) {
      this.audioThread.setBlockEnd( this.blockErr );
    }
    return status;
  }


  /*
   * Einlesen eines Blocks
   *
   * Bit 7 ist immer negiert.
   * Da die Pruefbyteberechnung mit dem negierten Bit arbeitet,
   * wird Bit 7 hier in der Methode nicht korrigiert,
   * beim Lesen des Blocks, wo auch die Pruefbyteberechnung stattfindet.
   */
  private int readByte()
  {
    int value = 0;

    // auf Stopp-Bit-Halbwelle warten
    int micros = this.audioThread.readMicrosTillPhaseChange();
    while( (micros >= 0) && (micros >= BIT_DISTINCT_MICROS) ) {
      micros = this.audioThread.readMicrosTillPhaseChange();
    }

    // Stoppbits lesen
    while( (micros >= 0) && (micros < BIT_DISTINCT_MICROS) ) {
      this.audioThread.fillUpRecognizedWave( WAVE_PILOT );
      micros = this.audioThread.readMicrosTillPhaseChange();
    }

    // zweite Halbwelle des Startbits lesen
    this.audioThread.readMicrosTillPhaseChange();
    this.audioThread.fillUpRecognizedWave( WAVE_DELIM );

    // 8 Datenbits lesen
    for( int i = 0; this.audioThread.isIOEnabled() && (i < 8); i++ ) {
      value >>= 1;
      micros = this.audioThread.readMicrosTillPhaseChange()
			+ this.audioThread.readMicrosTillPhaseChange();
      if( micros > (5 * BIT_DISTINCT_MICROS) ) {
	value = -1;
	break;
      }
      if( micros < (BIT_DISTINCT_MICROS + BIT_DISTINCT_MICROS) ) {
	value |= 0x80;
        this.audioThread.readMicrosTillPhaseChange();
        this.audioThread.readMicrosTillPhaseChange();
	this.audioThread.fillUpRecognizedWave( WAVE_BIT_1 );
      } else {
	this.audioThread.fillUpRecognizedWave( WAVE_BIT_0 );
      }
    }
    if( !this.audioThread.isIOEnabled() ) {
      value = -1;
    }
    return value;
  }


  private void writeEOL( OutputStream out ) throws IOException
  {
    if( this.eol != null ) {
      int len = this.eol.length();
      for( int i = 0; i < len; i++ ) {
	out.write( this.eol.charAt( i ) );
      }
    }
  }
}
