/*
 * (c) 2023-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Hilfsfunktionen fuer Audio
 */

package jkcload.audio;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.io.Closeable;
import java.io.IOException;
import javax.sound.sampled.AudioFileFormat;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.Control;
import javax.sound.sampled.FloatControl;
import javax.sound.sampled.Line;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.Mixer;
import javax.sound.sampled.SourceDataLine;
import javax.sound.sampled.TargetDataLine;


public class AudioUtil
{
  private static Map<String,AudioFileFormat.Type> ext2AudioFileType = null;


  public static void closeSilently( Closeable closeable )
  {
    if( closeable != null ) {
      try {
	closeable.close();
      }
      catch( IOException ex ) {}
    }
  }


  public static AudioFileFormat.Type getAudioFileTypeByExtension(
							String fileExt )
  {
    return fileExt != null ?
		getFileExtToAudioFileTypeMap().get( fileExt.toLowerCase() )
		:null;
  }


  public static Map<String,AudioFileFormat.Type>
					getFileExtToAudioFileTypeMap()
  {
    if( ext2AudioFileType == null ) {
      ext2AudioFileType = new HashMap<>();
      for( AudioFileFormat.Type t : AudioSystem.getAudioFileTypes() ) {
	String ext = t.getExtension();
	if( ext != null ) {
	  if( !ext.isEmpty() ) {
	    ext = ext.toLowerCase();
	    ext2AudioFileType.put( ext, t );
	  }
	}
      }
    }
    return ext2AudioFileType;
  }


  public static List<Mixer.Info> getSourceDataLineMixerInfo()
  {
    List<Mixer.Info> rv = new ArrayList<>();
    for( Mixer.Info mixerInfo : AudioSystem.getMixerInfo() ) {
      try {
	for( Line.Info lineInfo : AudioSystem.getMixer( mixerInfo )
						.getSourceLineInfo() )
	{
	  if( lineInfo instanceof SourceDataLine.Info ) {
	    rv.add( mixerInfo );
	    break;
	  }
	}
      }
      catch( IllegalArgumentException ex ) {}
    }
    return rv;
  }


  public static List<Mixer.Info> getTargetDataLineMixerInfo()
  {
    List<Mixer.Info> rv = new ArrayList<>();
    for( Mixer.Info mixerInfo : AudioSystem.getMixerInfo() ) {
      try {
	for( Line.Info lineInfo : AudioSystem.getMixer( mixerInfo )
						.getTargetLineInfo() )
	{
	  if( lineInfo instanceof TargetDataLine.Info ) {
	    rv.add( mixerInfo );
	    break;
	  }
	}
      }
      catch( IllegalArgumentException ex ) {}
    }
    return rv;
  }


  public static void closeSourceDataLine( SourceDataLine line )
  {
    if( line != null ) {
      try {
	line.stop();
	line.flush();
	line.close();
      }
      catch( Exception ex ) {}
    }
  }


  public static SourceDataLine openSourceDataLine(
					AudioFormat audioFmt,
					String      mixerName )
		throws IllegalArgumentException, LineUnavailableException
  {
    SourceDataLine line      = null;
    Mixer.Info     mixerInfo = null;
    if( mixerName != null ) {
      if( !mixerName.isEmpty() ) {
	for( Mixer.Info mInfo : AudioSystem.getMixerInfo() ) {
	  String mName = mInfo.getName();
	  if( mName != null ) {
	    if( mName.equals( mixerName ) ) {
	      mixerInfo = mInfo;
	      break;
	    }
	  }
	}
	if( mixerInfo == null ) {
	  throw new IllegalArgumentException( mixerName
				+ ":\nAudioger\u00E4t nicht gefunden" );
	}
      }
    }
    if( mixerInfo != null ) {
      line = AudioSystem.getSourceDataLine( audioFmt, mixerInfo );
    } else {
      line = AudioSystem.getSourceDataLine( audioFmt );
    }
    line.open( audioFmt );
    line.start();
    return line;
  }


  public static void setVolume( Line line, float volume )
  {
    if( line != null ) {
      FloatControl control = null;
      for( Control.Type t : new Control.Type[] {
					FloatControl.Type.MASTER_GAIN,
					FloatControl.Type.VOLUME } )
      {
	try {
	  if( line.isControlSupported( t ) ) {
	    Control c = line.getControl( t );
	    if( c instanceof FloatControl ) {
	      control = (FloatControl) c;
	      break;
	    }
	  }
	}
	catch( IllegalArgumentException ex ) {}
      }
      if( control != null ) {
	// logarithmische Lautstaerkeregelung
	volume = (float) Math.log( 1.0 + (volume * (Math.E - 1.0)) );
	try {
	  float minValue = control.getMinimum();
	  float maxValue = control.getMaximum();
	  float range    = maxValue - minValue;
	  if( range > 0F ) {
	    float value = minValue + (volume * range);
	    if( value < minValue ) {
	      value = minValue;
	    } else if( value > maxValue ) {
	      value = maxValue;
	    }
	    control.setValue( value );
	  }
	}
	catch( IllegalArgumentException ex ) {}
      }
    }
  }


	/* --- private Methoden --- */



	/* --- Konstruktor --- */

  private AudioUtil()
  {
    // leer
  }
}
