/*
 * (c) 2023-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Thread fuer die Audioverarbeitung
 */

package jkcload.audio;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import javax.sound.sampled.AudioFormat;
import jkcload.Main;


public abstract class AudioThread extends Thread
{
  public interface Observer
  {
    public void analysisData(
			AudioFormat audioFmt,
			byte[]      audioData,
			byte[]      recognitionData );

    public void audioStatusChanged(
				AudioThread audioThread,
				boolean     active,
				String      formatInfo,
				String      errMsg );

    public void errorOccured( String text, Exception ex );

    public void fileRead(
			String             format,
			int                begAddr,
			int                dataLen,
			int                startAddr,
			String             fileName,
			String             infoText,
			List<String>       fileExts,
			Map<String,byte[]> fileExt2Bytes,
			String             logText,
			boolean            errState );

    public void setVolumeLimits( int minValue, int maxValue );
    public void updProgressPercent( int percent );
    public void updReadActivity( String text );
    public void updVolume( int volume );
  };


  protected Observer observer;

  private RecognitionSettings      recognSettings;
  private volatile AudioProcessor  audioProcessor;
  private int                      selectedChannel;
  private int                      adjustPeriodCnt;
  private int                      adjustPeriodLen;
  private int                      channels;
  private int                      minValue;
  private int                      maxValue;
  private int                      previousValue;
  private int                      recentValue;
  private int                      recognBlockBegIdx;
  private int                      recognWaveIdx;
  private int                      sampleSize;
  private int                      sampleBitMask;
  private int                      sampleBitNegMask;
  private int                      sampleSignMask;
  private int                      frameSize;
  private float                    frameRate;
  private float                    minLevel;
  private float                    maxRange;
  private Float                    steepFlankStroke;
  private boolean                  bigEndian;
  private boolean                  dataSigned;
  private boolean                  recAnalysisData;
  private boolean                  lastPhase;
  private boolean                  phase;
  private volatile boolean         updVolumeEnabled;
  private volatile boolean         ioEnabled;
  private byte[]                   frameBuf;
  private String                   errorMsg;
  private AudioFormat              audioFmt;
  private ByteArrayOutputStream    audioOut;
  private ExtByteArrayOutputStream recognitionOut;


  protected AudioThread(
		int                 selectedChannel,
		RecognitionSettings recognSettings,
		boolean             recAnalysisData,
		boolean             updVolumeEnabled,
		Observer            observer )
  {
    super( Main.APPNAME + " Audio Processor" );
    this.selectedChannel   = selectedChannel;
    this.recognSettings    = recognSettings;
    this.updVolumeEnabled  = updVolumeEnabled;
    this.recAnalysisData   = recAnalysisData;
    this.observer          = observer;
    this.audioFmt          = null;
    this.audioProcessor    = null;
    this.minLevel          = recognSettings.getMinLevel();
    this.steepFlankStroke  = recognSettings.getSteepFlankStroke();
    this.frameRate         = 0F;
    this.frameSize         = 0;
    this.sampleSize        = 0;
    this.sampleBitMask     = 0;
    this.sampleBitNegMask  = 0;
    this.sampleSignMask    = 0;
    this.adjustPeriodCnt   = 0;
    this.adjustPeriodLen   = 0;
    this.channels          = 0;
    this.minValue          = Integer.MAX_VALUE;
    this.maxValue          = -Integer.MAX_VALUE;
    this.maxRange          = 0F;
    this.previousValue     = 0;
    this.recentValue       = 0;
    this.recognBlockBegIdx = -1;
    this.recognWaveIdx     = 0;
    this.bigEndian         = false;
    this.dataSigned        = false;
    this.lastPhase         = false;
    this.phase             = false;
    this.ioEnabled         = true;
    this.errorMsg          = null;
    this.frameBuf          = new byte[ 4 ];
    switch( recognSettings.getFormat() ) {
      case AC1:
	this.audioProcessor = new AC1AudioProcessor(
					this,
					recognSettings.getTolerance() );
	break;
      case BASICODE:
	this.audioProcessor = new BCAudioProcessor(
					this,
					recognSettings.getTolerance() );
	break;
      case BCS3_2_5MHz:
	this.audioProcessor = new BCS3AudioProcessor(
					this,
					recognSettings.getTolerance(),
					false );
	break;
      case BCS3_3_5MHz:
	this.audioProcessor = new BCS3AudioProcessor(
					this,
					recognSettings.getTolerance(),
					true );
	break;
      case KC:
	this.audioProcessor = new KCAudioProcessor(
					this,
					recognSettings.getTolerance() );
	break;
      case SCCH:
	this.audioProcessor = new SCCHAudioProcessor(
					this,
					recognSettings.getTolerance() );
	break;
      case Z1013_1MHz:
	this.audioProcessor = new Z1013AudioProcessor(
					this,
					recognSettings.getTolerance(),
					1 );
	break;
      case Z1013_2MHz:
	this.audioProcessor = new Z1013AudioProcessor(
					this,
					recognSettings.getTolerance(),
					2 );
	break;
      case Z1013_4MHz:
	this.audioProcessor = new Z1013AudioProcessor(
					this,
					recognSettings.getTolerance(),
					4 );
	break;
      case ZX_SPECTRUM:
	this.audioProcessor = new ZXAudioProcessor(
					this,
					recognSettings.getTolerance() );
	break;
    }
  }


  protected void fillUpRecognizedWave( int value )
  {
    fillUpRecognizedWave( this.recognWaveIdx, value );
  }


  protected void fillUpRecognizedWave( int fromIdx, int value )
  {
    if( (this.recognitionOut != null) && (fromIdx >= 0) ) {
      this.recognitionOut.orToEnd(
			fromIdx,
			value,
			AudioProcessor.WAVE_MASK );
      this.recognWaveIdx = this.recognitionOut.size();
    }
  }


  public void fireStop()
  {
    this.ioEnabled = false;
    interrupt();
  }


  protected abstract AudioFormat openAudioSource() throws IOException;
  protected abstract void        closeAudioSource();
  protected abstract int         readAudioData(
					byte[] buf,
					int    offs,
					int    len ) throws IOException;


  protected long getFrameLength()
  {
    return -1L;
  }


  public Observer getObserver()
  {
    return this.observer;
  }


  protected int getRecognitionIndex()
  {
    return this.recognitionOut != null ? this.recognitionOut.size() : -1;
  }


  public boolean isIOEnabled()
  {
    return this.ioEnabled;
  }


  public boolean isUpdVolumeEnabled()
  {
    return this.updVolumeEnabled;
  }


  protected void markBlockBegin()
  {
    if( this.recognitionOut != null )
      this.recognBlockBegIdx = this.recognitionOut.size();
  }


  public int readMicrosTillPhaseChange()
  {
    int micros = -1;
    if( this.frameRate > 0F ) {

      /*
       * Frames bis zum Phasenwechsel lesen
       *
       * Da beim vorherigen Aufruf die aktuelle Phase
       * bereits gelesen wurde, beginnt die Zaehlung mit 1.
       */
      int n = 1;
      while( this.ioEnabled
	     && (readFrameAndGetPhase() == this.lastPhase) )
      {
	n++;
      }
      this.lastPhase = !this.lastPhase;

      // Anzahl gelesener Frames im Mikrosekunden umrechnen
      micros = Math.round( 1000000F * (float) n / this.frameRate );
    }
    return micros;
  }


  protected void setBlockEnd( boolean errState )
  {
    if( (this.recognitionOut != null) && (this.recognBlockBegIdx >= 0) ) {
      int v = AudioProcessor.BLOCK_READ;
      if( errState ) {
	v |= AudioProcessor.BLOCK_ERROR;
      }
      this.recognitionOut.orToEnd( this.recognBlockBegIdx, v );
      this.recognBlockBegIdx = -1;
    }
  }


  public void setMonitorEnabled(
			boolean state,
			String  mixerName,
			float   volume )
  {
    // leer
  }


  public void setMonitorVolume( float volume )
  {
    // leer
  }


  public boolean supportsMonitoring()
  {
    return false;
  }


	/* --- Runnable --- */

  @Override
  public void run()
  {
    try {
      boolean     fmtOK            = false;
      int         sampleSizeInBits = 0;
      AudioFormat fmt              = openAudioSource();
      if( fmt != null ) {
	AudioFormat.Encoding encoding  = fmt.getEncoding();
	if( encoding != null ) {
	  if( encoding.equals( AudioFormat.Encoding.PCM_UNSIGNED ) ) {
	    this.dataSigned = false;
	    fmtOK           = true;
	  } else if( encoding.equals( AudioFormat.Encoding.PCM_SIGNED ) ) {
	    this.dataSigned = true;
	    fmtOK           = true;
	  }
	}
	if( fmtOK ) {
	  this.frameRate  = fmt.getFrameRate();
	  this.channels   = fmt.getChannels();
	  if( this.frameRate < 1F ) {
	    fmtOK = false;
	  }
	  if( this.channels < 1 ) {
	    fmtOK = false;
	  }
	  if( (this.selectedChannel < 0)
	      || (this.selectedChannel >= this.channels) )
	  {
	    this.selectedChannel = 0;
	  }
	  sampleSizeInBits = fmt.getSampleSizeInBits();
	  this.sampleSize  = (sampleSizeInBits + 7) / 8;
	  this.frameSize   = fmt.getFrameSize();
	  if( (sampleSizeInBits < 1)
	      || (sampleSizeInBits > 24)
	      || (this.sampleSize < 1)
	      || (this.frameSize < 1)
	      || (this.frameSize < (this.sampleSize * this.channels)) )
	  {
	    fmtOK = false;
	  }
	  this.sampleBitMask    = ((1 << sampleSizeInBits) - 1);
	  this.sampleBitNegMask = ~this.sampleBitMask;
	  this.sampleSignMask   = (1 << (sampleSizeInBits - 1));
	  this.bigEndian        = fmt.isBigEndian();
	  this.maxRange         = (float) this.sampleBitMask;
	}
      }
      if( !fmtOK ) {
	throw new IOException( "Audioformat nicht unterst\u00FCtzt" );
      }
      if( this.frameBuf.length < this.frameSize ) {
	this.frameBuf = new byte[ this.frameSize ];
      }
      StringBuilder buf = new StringBuilder( 256 );
      buf.append( Math.round( this.frameRate ) );
      buf.append( " Hz, " );
      buf.append( sampleSizeInBits );
      buf.append( " Bit" );
      if( this.channels == 1 ) {
	buf.append( " Mono" );
      } else if( this.channels == 2 ) {
	buf.append( " Stereo" );
      } else {
	buf.append( ", " );
	buf.append( this.channels );
	buf.append( " Kan\u00E4le" );
      }
      long frameLen = getFrameLength();
      if( frameLen > 0 ) {
	int seconds = Math.round( (float) frameLen / this.frameRate );
	int minutes = seconds / 60;
	int hours   = minutes / 60;
	minutes %= 60;
	seconds %= 60;
	buf.append( String.format(
			", L\u00E4nge: %02d:%02d:%02d",
			hours,
			minutes,
			seconds ) );
      }
      this.observer.audioStatusChanged(
				this,
				true,
				buf.toString(),
				null );

      // ggf. Puffer fuer Analysedaten anlegen
      if( this.recAnalysisData ) {
	this.audioFmt = new AudioFormat(
				this.frameRate,
				sampleSizeInBits,
				1,
				this.dataSigned,
				this.bigEndian );
	this.audioOut       = new ByteArrayOutputStream( 0x8000 );
	this.recognitionOut = new ExtByteArrayOutputStream( 0x8000 );
      }

      // Wertebereich der Pegelanzeige festlegen
      if( this.sampleSize == 1 ) {
	if( this.dataSigned ) {
	  this.observer.setVolumeLimits( -128, 127 );
	} else {
	  this.observer.setVolumeLimits( 0, 255 );
	}
      } else {
	if( this.dataSigned ) {
	  this.observer.setVolumeLimits( -32768, 32767 );
	} else {
	  this.observer.setVolumeLimits( 0, 65535 );
	}
      }

      /*
       * Min-/Max-Regelung initialisieren
       *
       * Nach einer Periodenlaenge werden die Minimum- und Maximum-Werte
       * zueinander um einen Schritt angenaehert,
       * um so einen dynamischen Mittelwert errechnen zu koennen.
       */
      this.adjustPeriodLen = Math.round( this.frameRate ) / 256;
      if( this.adjustPeriodLen < 1 ) {
	this.adjustPeriodLen = 1;
      }
      this.adjustPeriodCnt = this.adjustPeriodLen;

      // Audiodaten lesen und verarbeiten
      this.lastPhase = readFrameAndGetPhase();
      do {
	AudioProcessor audioProcessor = this.audioProcessor;
	if( audioProcessor == null ) {
	  break;
	}
	audioProcessor.run();
      } while( this.ioEnabled );
    }
    catch( IOException ex ) {
      this.errorMsg = ex.getMessage();
    }
    catch( Exception ex ) {}
    finally {
      closeAudioSource();
      this.observer.audioStatusChanged(
				this,
				false,
				null,
				this.errorMsg );
      if( (this.audioFmt != null)
	  && (this.audioOut != null)
	  && (this.recognitionOut != null) )
      {
	int audioLen       = this.audioOut.size();
	int recognitionLen = this.recognitionOut.size();
	if( (audioLen > 0) && (recognitionLen > 0) ) {
	  this.observer.analysisData(
				this.audioFmt,
				this.audioOut.toByteArray(),
				this.recognitionOut.toByteArray() );
	}
      }
    }
  }


	/* --- private Methoden --- */

  /*
   * Die Methode liest ein Frame und gibt die Phasenlage
   * des ausgewaehlten Kanals zurueck.
   */
  private boolean readFrameAndGetPhase()
  {
    boolean phase = this.phase;
    try {
      int v = readFrameAndGetSample();
      if( v >= 0 ) {
	v &= this.sampleBitMask;

	// Wenn gelesener Wert negativ, dann Zahl korrigieren
	if( this.dataSigned && ((v & this.sampleSignMask) != 0) ) {
	  v |= this.sampleBitNegMask;
	}

	// Minimum-/Maximum-Werte anpassen
	if( this.adjustPeriodCnt > 0 ) {
	  --this.adjustPeriodCnt;
	} else {
	  this.adjustPeriodCnt = this.adjustPeriodLen;
	  if( this.minValue < this.maxValue ) {
	    this.minValue++;
	  }
	  if( this.maxValue > this.minValue ) {
	    --this.maxValue;
	  }
	}
	if( v < this.minValue ) {
	  this.minValue = v;
	}
	else if( v > this.maxValue ) {
	  this.maxValue = v;
	}
	if( this.updVolumeEnabled ) {
	  this.observer.updVolume( v );
	}
	int range = this.maxValue - this.minValue;
	if( range > Math.round( this.minLevel * this.maxRange ) ) {
	  if( this.steepFlankStroke != null ) {

	    // Phasenerkennung anhand steiler Flanken
	    int absFlankStroke = Math.round(
		  (float) range * this.steepFlankStroke.floatValue() );
	    if( phase ) {
	      if( (this.previousValue >= this.recentValue)
		  && (this.previousValue > (v + absFlankStroke)) )
	      {
		phase = false;
	      }
	    } else {
	      if( (this.previousValue <= this.recentValue)
		  && (this.previousValue < (v - absFlankStroke)) )
	      {
		phase = true;
	      }
	    }
	    this.previousValue = this.recentValue;
	    this.recentValue   = v;

	  } else {

	    // Phasenerkennung anhand des dynamischen Mittelwertes
	    if( this.phase ) {
	      if( v < (this.minValue + (range * 48 / 100)) ) {
		phase = false;
	      }
	    } else {
	      if( v > (this.minValue + (range * 52 / 100)) ) {
		phase = true;
	      }
	    }
	  }
	  this.phase = phase;
	}
	if( this.recognitionOut != null ) {
	  this.recognitionOut.write( phase ? AudioProcessor.PHASE_MASK : 0 );
	}
      } else {
	this.ioEnabled = false;
      }
    }
    catch( Exception ex ) {
      /*
       * z.B. InterruptedException bei Programmbeendigung oder
       * eine andere Exception bei Abziehen eines aktiven USB-Audiogeraetes
       */
      this.ioEnabled = false;
    }
    return phase;
  }


  /*
   * Die Methode liest ein Frame und gibt das Sample
   * des ausgewaehlten Kanals zurueck.
   * Auch bei vorzeichenbehaftenen Audiodaten ist der Rueckgabewert
   * nicht negativ, da nur die betreffenden unteren Bits gefuellt sind.
   * Bei einem Rueckgabewert kleiner Null konnte kein Sample gelesen werden.
   */
  private int readFrameAndGetSample() throws IOException
  {
    int value = -1;
    do {
      int n = readAudioData( this.frameBuf, 0, this.frameSize );
      if( n < 0 ) {
	this.ioEnabled = false;
	break;
      }
      if( n == 0 ) {
	continue;
      }
      int offset = this.selectedChannel * this.sampleSize;
      if( (offset < 0)
	  || (offset > (this.frameBuf.length - this.sampleSize)) )
      {
	offset = 0;
      }
      value = 0;
      if( this.bigEndian ) {
	for( int i = 0; i < this.sampleSize; i++ ) {
	  value = (value << 8) | ((int) this.frameBuf[ offset + i ] & 0xFF);
	}
      } else {
	for( int i = this.sampleSize - 1; i >= 0; --i ) {
	  value = (value << 8) | ((int) this.frameBuf[ offset + i ] & 0xFF);
	}
      }
      if( this.audioOut != null ) {
	for( int i = 0; i < this.sampleSize; i++ ) {
	  this.audioOut.write( (int) this.frameBuf[ offset + i ] & 0xFF );
	}
      }
    }
    while( false );
    return value;
  }
}
