/*
 * (c) 2023-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Verarbeitung der Audiodaten
 */

package jkcload.audio;

import java.io.IOException;
import java.io.OutputStream;


public abstract class AudioProcessor
{
  public static final int PHASE_MASK  = 0x80;
  public static final int BLOCK_READ  = 0x40;
  public static final int BLOCK_ERROR = 0x20;
  public static final int WAVE_MASK   = 0x07;
  public static final int WAVE_PILOT  = 0x01;
  public static final int WAVE_DELIM  = 0x02;
  public static final int WAVE_BIT_0  = 0x04;
  public static final int WAVE_BIT_1  = 0x05;
  public static final int NONE        = 0x00;

  protected static final String CKS_ERROR = "Pr\u00FCfsummenfehler";

  protected AudioThread          audioThread;
  protected AudioThread.Observer observer;


  protected AudioProcessor( AudioThread audioThread )
  {
    this.audioThread = audioThread;
    this.observer    = audioThread.getObserver();
  }


  protected static void appendColonChecksumErrorTo( StringBuilder buf )
  {
    buf.append( ": " );
    buf.append( CKS_ERROR );
  }


  protected static void appendIncompleteReadTo(
					StringBuilder buf,
					int           missingBytes )
  {
    buf.append( "Unvollst\u00E4ndig eingelesen" );
    if( missingBytes == 1 ) {
      buf.append( ", 1 Byte fehlt" );
    } else if( missingBytes > 1 ) {
      buf.append( ", " );
      buf.append( missingBytes );
      buf.append( " Bytes fehlen" );
    }
    buf.append( '\n' );
  }


  protected static int getWord( byte[] buf, int idx )
  {
    int rv = 0;
    if( (idx >= 0) && ((idx + 1) < buf.length) ) {
      rv = (buf[ idx ] & 0x00FF) | ((buf[ idx + 1 ] << 8) & 0xFF00);
    }
    return rv;
  }


  public abstract void run();


  protected static void writeHeadersaveHeader(
				OutputStream out,
				int          begAddr,
				int          endAddr,
				int          startAddr,
				int          fType,
				String       fName ) throws IOException
  {
    out.write( begAddr );
    out.write( begAddr >> 8 );
    out.write( endAddr );
    out.write( endAddr >> 8 );
    out.write( startAddr );
    out.write( startAddr >> 8 );
    for( int i = 0; i < 6; i++ ) {
      out.write( 0 );
    }
    out.write( fType );
    for( int i = 0; i < 3; i++ ) {
      out.write( 0xD3 );
    }
    writeFixLengthASCII( out, fName, 16 );
  }


	/* --- privateMethoden --- */

  private static void writeFixLengthASCII(
				OutputStream out,
				String       text,
				int          nChars ) throws IOException
  {
    if( text != null ) {
      int len = text.length();
      int idx = 0;
      while( (nChars > 0) && (idx < len) ) {
	char ch = text.charAt( idx );
	if( (ch < '\u0020') || (ch > '\u007E') ) {
	  ch = '\u0020';
	}
	out.write( ch );
	--nChars;
	idx++;
      }
    }
    while( nChars > 0 ) {
      out.write( 0x20 );
      --nChars;
    }
  }
}
