/*
 * (c) 2023-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Thread fuer einen Audiokanal
 */

package jkcload.audio;

import java.io.IOException;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.Mixer;
import javax.sound.sampled.TargetDataLine;


public class AudioLineThread extends AudioThread
{
  private Mixer.Info     mixerInfo;
  private TargetDataLine line;
  private Exception      exception;


  public AudioLineThread(
		Mixer.Info          mixerInfo,
		int                 channel,
		RecognitionSettings recognSettings,
		boolean             recAnalysisData,
		Observer            observer )
  {
    super( channel, recognSettings, recAnalysisData, true, observer );
    this.mixerInfo = mixerInfo;
    this.line      = null;
    this.exception = null;
  }


	/* --- ueberschriebene Methoden --- */

  @Override
  protected void closeAudioSource()
  {
    if( this.line != null ) {
      try {
	this.line.stop();
	this.line.flush();
      }
      finally {
	this.line.close();
	this.line = null;
      }
    }
  }


  @Override
  protected AudioFormat openAudioSource() throws IOException
  {
    this.line = openTargetDataLine( this.mixerInfo, 44100F );
    if( this.line == null ) {
      String msg = "Audiokanal kann nicht ge\u00F6ffnet werden.";
      if( this.exception != null ) {
	String exMsg = this.exception.getMessage();
	if( exMsg != null ) {
	  exMsg = exMsg.trim();
	  if( !exMsg.isEmpty() ) {
	    msg = msg + "\n\n" + exMsg;
	  }
	}
      }
      throw new IOException( msg );
    }
    this.line.start();
    return this.line.getFormat();
  }



  @Override
  public int readAudioData(
			byte[] buf,
			int    offs,
			int    len ) throws IOException
  {
    int rv = -1;
    if( this.line != null ) {
      try {
	rv = this.line.read( buf, offs, len );
	if( rv == 0 ) {
	  rv = -1;
	}
      }
      catch( IllegalArgumentException ex ) {
	throw new IOException( ex.getMessage() );
      }
    }
    return rv;
  }


	/* --- private Methoden --- */

  private TargetDataLine openTargetDataLine(
				Mixer.Info mixerInfo,
				float      sampleRate )
  {
    TargetDataLine line = openTargetDataLine( mixerInfo, sampleRate, 16 );
    if( line == null ) {
      line = openTargetDataLine( mixerInfo, sampleRate, 8 );
    }
    return line;
  }


  private TargetDataLine openTargetDataLine(
				Mixer.Info mixerInfo,
				float      sampleRate,
				int        sampleSizeInBits )
  {
    TargetDataLine line = openTargetDataLine(
				mixerInfo,
				sampleRate,
				sampleSizeInBits,
				2 );
    if( line == null ) {
      line = openTargetDataLine(
				mixerInfo,
				sampleRate,
				sampleSizeInBits,
				1 );
    }
    return line;
  }


  private TargetDataLine openTargetDataLine(
				Mixer.Info mixerInfo,
				float      sampleRate,
				int        sampleSizeInBits,
				int        channels )
  {
    boolean dataSigned = (sampleSizeInBits > 8);
    TargetDataLine line = openTargetDataLine(
				mixerInfo,
				sampleRate,
				sampleSizeInBits,
				channels,
				dataSigned );
    if( line == null ) {
      line = openTargetDataLine(
				mixerInfo,
				sampleRate,
				sampleSizeInBits,
				channels,
				!dataSigned );
    }
    return line;
  }


  private TargetDataLine openTargetDataLine(
				Mixer.Info mixerInfo,
				float      sampleRate,
				int        sampleSizeInBits,
				int        channels,
				boolean    dataSigned )
  {
    TargetDataLine line = openTargetDataLine(
				mixerInfo,
				sampleRate,
				sampleSizeInBits,
				channels,
				dataSigned,
				false );
    if( line == null ) {
      line = openTargetDataLine(
				mixerInfo,
				sampleRate,
				sampleSizeInBits,
				channels,
				dataSigned,
				true );
    }
    return line;
  }


  private TargetDataLine openTargetDataLine(
				Mixer.Info mixerInfo,
				float      sampleRate,
				int        sampleSizeInBits,
				int        channels,
				boolean    dataSigned,
				boolean    bigEndian )
  {
    TargetDataLine line = null;
    try {
      AudioFormat fmt = new AudioFormat(
				sampleRate,
				sampleSizeInBits,
				channels,
				dataSigned,
				bigEndian );
      if( mixerInfo != null ) {
	line = AudioSystem.getTargetDataLine( fmt, mixerInfo );
      } else {
	line = AudioSystem.getTargetDataLine( fmt );
      }
      line.open( fmt );
    }
    catch( Exception ex ) {
      if( this.exception == null ) {
	this.exception = ex;
      }
      line = null;
    }
    return line;
  }
}
