/*
 * (c) 2023-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Thread fuer eine Audiodatei
 */

package jkcload.audio;

import java.io.File;
import java.io.IOException;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.SourceDataLine;
import javax.sound.sampled.UnsupportedAudioFileException;


public class AudioFileThread extends AudioThread
{
  public static final String TEXT_UNSUPPORTED_FILE_FORMAT
				= "Dateiformat nicht unterst\u00FCtzt";

  private File             file;
  private AudioInputStream in;
  private SourceDataLine   monitorLine;
  private Object           monitorLock;
  private Boolean          requestedMonitorState;
  private String           requestedMonitorMixerName;
  private Float            requestedMonitorVolume;
  private long             frameLen;
  private long             byteLen;
  private long             bytesRead;
  private int              percent;


  public AudioFileThread(
		File                file,
		int                 channel,
		RecognitionSettings recognSettings,
		boolean             monitorEnabled,
		String              monitorMixerName,
		float               monitorVolume,
		boolean             recAnalysisData,
		Observer            observer )
  {
    super( channel, recognSettings, recAnalysisData, false, observer );
    this.file                      = file;
    this.requestedMonitorState     = Boolean.valueOf( monitorEnabled );
    this.requestedMonitorMixerName = monitorMixerName;
    this.requestedMonitorVolume    = Float.valueOf( monitorVolume );
    this.in                        = null;
    this.monitorLine               = null;
    this.monitorLock               = new Object();
    this.frameLen                  = -1L;
    this.byteLen                   = 0;
    this.bytesRead                 = 0;
    this.percent                   = 0;
  }


	/* --- ueberschriebene Methoden --- */

  @Override
  protected void closeAudioSource()
  {
    closeMonitorLine();
    if( this.in != null ) {
      try {
	this.in.close();
      }
      catch( IOException ex ) {}
      finally {
	this.in = null;
      }
    }
  }


  @Override
  protected long getFrameLength()
  {
    return this.frameLen;
  }


  @Override
  protected AudioFormat openAudioSource() throws IOException
  {
    try {
      this.in       = AudioSystem.getAudioInputStream( file );
      this.frameLen = this.in.getFrameLength();
    }
    catch( UnsupportedAudioFileException ex ) {
      throw new IOException( TEXT_UNSUPPORTED_FILE_FORMAT );
    }
    AudioFormat fmt = this.in.getFormat();
    if( this.frameLen > 0 ) {
      this.byteLen   = this.frameLen * fmt.getFrameSize();
      this.bytesRead = 0;
    }
    return fmt;
  }


  @Override
  public int readAudioData(
			byte[] buf,
			int    offs,
			int    len ) throws IOException
  {
    int nRead = -1;
    if( this.in != null ) {
      nRead = this.in.read( buf, offs, len );
      if( nRead > 0 ) {
	Boolean requestedMonitorState  = null;
	String  requestedMonitorMixerName = null;
	Float   requestedMonitorVolume = null;
	synchronized( this.monitorLock ) {
	  requestedMonitorState          = this.requestedMonitorState;
	  requestedMonitorMixerName      = this.requestedMonitorMixerName;
	  requestedMonitorVolume         = this.requestedMonitorVolume;
	  this.requestedMonitorState     = null;
	  this.requestedMonitorMixerName = null;
	  this.requestedMonitorVolume    = null;
	}
	if( requestedMonitorState != null ) {
	  if( requestedMonitorState.booleanValue()
	      && (this.monitorLine == null) )
	  {
	    openMonitorLine( requestedMonitorMixerName );
	  }
	}
	if( (this.monitorLine != null) && (requestedMonitorState != null) ) {
	  if( !requestedMonitorState.booleanValue() ) {
	    closeMonitorLine();
	  }
	}
	if( this.monitorLine != null ) {
	  if( requestedMonitorVolume != null ) {
	    AudioUtil.setVolume(
			this.monitorLine,
			requestedMonitorVolume.floatValue() );
	  }
	  this.monitorLine.write( buf, offs, Math.min( nRead, len ) );
	}
	this.bytesRead += nRead;
	int percent = (int) (this.bytesRead * 100L / this.byteLen);
	if( percent > 100 ) {
	  percent = 100;
	}
	if( percent > this.percent ) {
	  this.percent = percent;
	  this.observer.updProgressPercent( percent );
	}
      }
    }
    return nRead;
  }


  @Override
  public void setMonitorEnabled(
			boolean state,
			String  mixerName,
			float   volume )
  {
    synchronized( this.monitorLock ) {
      this.requestedMonitorState     = state;
      this.requestedMonitorMixerName = mixerName;
      this.requestedMonitorVolume    = volume;
    }
  }


  @Override
  public void setMonitorVolume( float volume )
  {
    synchronized( this.monitorLock ) {
      this.requestedMonitorVolume = volume;
    }
  }


  @Override
  public boolean supportsMonitoring()
  {
    return true;
  }


	/* --- private Methoden --- */

  private void closeMonitorLine()
  {
    if( this.monitorLine != null ) {
      try {
	AudioUtil.closeSourceDataLine( this.monitorLine );
      }
      finally {
	this.monitorLine = null;
      }
    }
  }


  private void openMonitorLine( String requestedMixerName )
  {
    try {
      AudioInputStream in = this.in;
      if( (in != null) && (this.monitorLine == null) ) {
	AudioFormat inFmt = in.getFormat();
	if( inFmt != null ) {
	  int sampleSizeInBits   = inFmt.getSampleSizeInBits();
	  AudioFormat monitorFmt = new AudioFormat(
					inFmt.getEncoding(),
					inFmt.getSampleRate(),
					sampleSizeInBits,
					1,
					(sampleSizeInBits + 7) / 8,
					inFmt.getFrameRate(),
					false );
	  this.monitorLine = AudioUtil.openSourceDataLine(
						monitorFmt,
						requestedMixerName );
	}
      }
    }
    catch( Exception ex ) {
      this.observer.errorOccured(
			"Mith\u00F6hren nicht m\u00F6glich",
			ex );
    }
  }
}
