/*
 * (c) 2024-2025 Jens Mueller
 *
 * JKCLOAD
 *
 * Verarbeitung der Audiodaten im AC1-Format
 */

package jkcload.audio;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


public class AC1AudioProcessor extends PhaseModAudioProcessor
{
  private static final String AC1_BASIC = "AC1 BASIC";
  private static final String AC1_MC    = "AC1 MC";
  private static final String AC1_RAW   = "AC1 Rohdaten";

  private static final int HALF_BIT_MICROS = 333;	// halbe Bit-Laenge

  private StringBuilder         logBuf;
  private ByteArrayOutputStream dataBytes;
  private ByteArrayOutputStream rawBytes;
  private int                   begAddr;
  private int                   curAddr;
  private boolean               fileErr;


  public AC1AudioProcessor(
			AudioThread audioThread,
			float       tolerance )
  {
    super( audioThread, tolerance, HALF_BIT_MICROS, HALF_BIT_MICROS );
    this.dataBytes = new ByteArrayOutputStream( 0x4000 );
    this.rawBytes  = new ByteArrayOutputStream( 0x4000 );
    this.logBuf    = new StringBuilder();
    this.begAddr   = -1;
    this.curAddr   = -1;
    this.fileErr   = false;
    reset();
  }


	/* --- ueberschrieben Methoden --- */

  @Override
  protected int readByte()
  {
    int b = super.readByte();
    if( b >= 0 ) {
      this.rawBytes.write( b );
    }
    return b;
  }


  @Override
  public void run()
  {
    try {
      this.audioThread.fillUpRecognizedWave( NONE );
      this.logBuf.setLength( 0 );
      this.dataBytes.reset();
      this.rawBytes.reset();
      this.begAddr = -1;
      this.curAddr = -1;
      this.fileErr = false;

      if( waitForSyncByte() ) {
	this.audioThread.markBlockBegin();

	// 7 Bytes lesen
	byte[] header = new byte[ 7 ];
	int    idx    = 0;
	while( idx < header.length ) {
	  int b = readByte();
	  if( b < 0 ) {
	    break;
	  }
	  header[ idx++ ] = (byte) b;
	}
	if( idx > 0 ) {

	  // Dateityp ermitteln
	  boolean done = false;
	  if( idx == header.length ) {
	    if( header[ 6 ] == (byte) 0xD3 ) {
	      done = readBasicFile( header );
	    } else if( header[ 0 ] == (byte) 0x55 ) {
	      done = readMCFile( header );
	    }
	  }
	  if( !done ) {

	    // Weder BASIC- noch MC-Datei -> alle Bytes als RAW-File lesen
	    this.observer.updReadActivity( AC1_RAW );
	    while( (this.rawBytes.size() < 0x20000) && (readByte() >= 0) )
	      ;	// readByte() schreibt Byte in this.rawBytes
	    this.observer.fileRead(
			AC1_RAW,
			-1,
			this.rawBytes.size(),
			-1,
			null,
			"Rohdaten der physischen Ebene",
			Collections.singletonList( "RAW" ),
			Collections.singletonMap(
					"RAW",
					this.rawBytes.toByteArray() ),
			null,
			false );
	  }
	}
      }
      this.observer.updReadActivity( null );
    }
    catch( Exception ex ) {
      this.observer.errorOccured( null, ex );
    }
  }


	/* --- private Methoden --- */

  private boolean readBasicFile( byte[] header )
  {
    boolean done = false;

    // Programmname
    StringBuilder nameBuf = new StringBuilder();
    for( int i = 0; i < 6; i++ ) {
      if( i < header.length ) {
	int b = header[ i ] & 0xFF;
	if( (b < 0x20) || (b > 0x7E) ) {
	  b = 0x20;
	}
	nameBuf.append( (char) b );
      }
    }
    String fileName = nameBuf.toString();

    // Aktivitaetsanzeige
    StringBuilder activityBuf = new StringBuilder( 64 );
    activityBuf.append( AC1_BASIC );
    activityBuf.append( ": " );
    activityBuf.append( fileName );
    String activity = activityBuf.toString();
    this.observer.updReadActivity( activity );

    // BASIC-Program einlesen
    int hLen = readByte();
    int lLen = readByte();
    if( (hLen >= 0) && (lLen >= 0) ) {
      int len = ((hLen << 8) & 0xFF00) | (lLen & 0x00FF);
      while( len > 0 ) {
	int b = readByte();
	if( b < 0 ) {
	  break;
	}
	this.dataBytes.write( b );
	--len;
      }
      if( len > 0 ) {
	this.fileErr = true;
	appendIncompleteReadTo( this.logBuf, len );
      }
      this.audioThread.setBlockEnd( this.fileErr );
      this.observer.fileRead(
			AC1_BASIC,
			-1,
			this.dataBytes.size(),
			-1,
			fileName,
			null,
			Collections.singletonList( "BAS" ),
			Collections.singletonMap(
					"BAS",
					this.dataBytes.toByteArray() ),
			this.logBuf.toString(),
			this.fileErr );
      done = true;
    }
    return done;
  }


  private boolean readMCDataBlock()
  {
    boolean state    = false;
    boolean blockErr = false;

    int len = readByte();
    if( len == 0 ) {
      len = 0x100;
    }
    int lAddr = readByte();
    int hAddr = readByte();
    if( (len >= 0) && (lAddr >= 0) && (hAddr >= 0) ) {
      int cks          = lAddr + hAddr;
      int expectedAddr = -1;
      int blockAddr    = ((hAddr << 8) & 0xFF00) | (lAddr & 0x00FF);
      this.observer.updReadActivity(
		String.format( "%s: %04Xh", AC1_MC, blockAddr ) );
      this.logBuf.append( String.format( "Block %04Xh", blockAddr ) );
      if( this.begAddr < 0 ) {
	this.begAddr = blockAddr;
	this.curAddr = blockAddr;
      } else if( blockAddr != this.curAddr ) {
	expectedAddr = this.curAddr;
	this.fileErr = true;
	this.curAddr = blockAddr;
      }
      int b = 0;
      while( len > 0 ) {
	b = readByte();
	if( b < 0 ) {
	  break;
	}
	this.dataBytes.write( b );
	cks += b;
	this.curAddr++;
	--len;
      }
      boolean errState = false;
      b                = readByte();
      if( b >= 0 ) {
	if( b != (cks & 0xFF) ) {
	  blockErr     = true;
	  this.fileErr = true;
	  appendColonChecksumErrorTo( this.logBuf );
	}
      }
      this.logBuf.append( '\n' );
      if( expectedAddr >= 0 ) {
	this.logBuf.append(
		String.format( "  Block %04Xh erwartet\n", expectedAddr ) );
      }
      this.audioThread.setBlockEnd( blockErr );
      if( len > 0 ) {
	appendIncompleteReadTo( this.logBuf, len );
      }
      state = true;
    }
    return state;
  }


  private boolean readMCFile( byte[] header )
  {
    boolean state = false;
    int     b     = 0;

    // Dateiname
    StringBuilder nameBuf = new StringBuilder();
    for( int i = 1; i < 17; i++ ) {
      if( i < header.length ) {
	b = header[ i ] & 0xFF;
      } else {
	b = readByte();
	if( b < 0 ) {
	  break;
	}
      }
      if( (b >= 0x20) && (b < 0x7F) ) {
	nameBuf.append( (char) b );
      } else {
	nameBuf.append( '_' );
      }
    }
    String fileName = nameBuf.toString();
    this.audioThread.setBlockEnd( false );

    // Aktivitaetsanzeige
    StringBuilder activityBuf = new StringBuilder( 64 );
    activityBuf.append( AC1_MC );
    activityBuf.append( ": " );
    activityBuf.append( fileName );
    String activity = activityBuf.toString();
    this.observer.updReadActivity( activity );

    // 256 Nullbytes, werden hier fehlertolerant eingelesen
    b = readByte();
    while( b == 0 ) {
      b = readByte();
    }

    // Bloecke lesen
    int startAddr = -1;
    do {
      this.audioThread.markBlockBegin();
      if( b == 0x3C ) {
	if( !readMCDataBlock() ) {
	  break;
	}
	state = true;
      } else if( b == 0x78 ) {
	int lAddr = readByte();
	int hAddr = readByte();
	if( (lAddr >= 0) && (hAddr >= 0) ) {
	  startAddr = ((hAddr << 8) & 0xFF00) | (lAddr & 0x00FF);
	  this.audioThread.setBlockEnd( false );
	}
	break;
      }
      b = readByte();
    } while( b >= 0 );
    if( state ) {
      int dataLen = this.dataBytes.size();
      if( dataLen > 0 ) {
	List<String>       fileExtList   = new ArrayList<>();
	Map<String,byte[]> fileExt2Bytes = new HashMap<>();
	try {
	  ByteArrayOutputStream hsBytes = new ByteArrayOutputStream(
							dataLen + 32 );
	  int endAddr = this.begAddr + dataLen - 1;
	  int fType   = 'M';
	  if( (startAddr >= begAddr) && (startAddr < endAddr) ) {
	    fType = 'C';
	  }
	  writeHeadersaveHeader(
			hsBytes,
			this.begAddr,
			endAddr,
			startAddr,
			fType,
			fileName );
	  this.dataBytes.writeTo( hsBytes );
	  fileExtList.add( "Z80" );
	  fileExt2Bytes.put( "Z80", hsBytes.toByteArray() );
	}
	catch( IOException ex ) {}
	fileExtList.add( "BIN" );
	fileExt2Bytes.put( "BIN", this.dataBytes.toByteArray() );
	this.observer.fileRead(
			AC1_MC,
			this.begAddr,
			dataLen,
			startAddr,
			fileName,
			null,
			fileExtList,
			fileExt2Bytes,
			this.logBuf.toString(),
			this.fileErr );
      } else {
	state = false;
      }
    }
    return state;
  }
}
